package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ListUtils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2016. 12. 8..
 */
public class ChatAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder>
    implements ChatAdapterContract.Model, ChatAdapterContract.View {

  private SortedList<MessageItem> items;

  @Nullable
  private OnTextMessageClickListener onTextMessageClickListener;
  @Nullable
  private OnFileMessageClickListener onFileMessageClickListener;
  @Nullable
  private OnImageMessageClickListener onImageMessageClickListener;
  @Nullable
  private OnWebMessageClickListener onWebMessageClickListener;
  @Nullable
  private OnSendingMessageClickListener onSendingMessageClickListener;
  @Nullable
  private OnProfileUpdateListener onProfileUpdateListener;
  @Nullable
  private OnFormMessageClickListener onFormMessageClickListener;

  @Nullable
  private WelcomeMessageItem welcomeMessageItem = null;

  private boolean isNewChat;
  private boolean hasChatEditTextFocus = false;

  public ChatAdapter(boolean isNewChat) {
    this.isNewChat = isNewChat;
    items = new SortedList<>(MessageItem.class, new SortedListCallback<MessageItem>(this, false));

    // Typing layout must always exists
    items.add(new TypingItem());
  }


  @Override
  public void setOnTextMessageClickListener(OnTextMessageClickListener onTextMessageClickListener) {
    this.onTextMessageClickListener = onTextMessageClickListener;
  }

  @Override
  public void setOnFileMessageClickListener(OnFileMessageClickListener onFileMessageClickListener) {
    this.onFileMessageClickListener = onFileMessageClickListener;
  }

  @Override
  public void setOnImageMessageClickListener(OnImageMessageClickListener onImageMessageClickListener) {
    this.onImageMessageClickListener = onImageMessageClickListener;
  }

  @Override
  public void setOnWebMessageClickListener(OnWebMessageClickListener onWebMessageClickListener) {
    this.onWebMessageClickListener = onWebMessageClickListener;
  }

  @Override
  public void setOnSendingMessageClickListener(OnSendingMessageClickListener onSendingMessageClickListener) {
    this.onSendingMessageClickListener = onSendingMessageClickListener;
  }

  @Override
  public void setOnProfileUpdateListener(OnProfileUpdateListener onProfileUpdateListener) {
    this.onProfileUpdateListener = onProfileUpdateListener;
  }

  @Override
  public void setOnFormMessageClickListener(OnFormMessageClickListener onFormMessageClickListener) {
    this.onFormMessageClickListener = onFormMessageClickListener;
  }

  @Override
  public void addMessage(Message message, @Nullable String chatId) {
    addMessages(Collections.singletonList(message), chatId);
  }

  @Override
  public void addMessages(@Nullable List<Message> messages, @Nullable String chatId) {
    if (!ListUtils.hasItems(messages) || TextUtils.isEmpty(chatId)) {
      return;
    }

    items.beginBatchedUpdates();

    for (Message message : messages) {
      if (message == null || !CompareUtils.isSame(chatId, message.getChatId())) {
        continue;
      }

      if (isNewChat &&
          message.getBotOption() != null &&
          message.getBotOption().isWelcome() &&
          welcomeMessageItem != null) {
        items.remove(welcomeMessageItem);
        items.add(new DateItem(message.getCreatedAt()));
        items.add(new ChatMessageItem(message));

        welcomeMessageItem = null;

        continue;
      }

      if (message.getLog() != null) {
        items.add(new LogMessageItem(message));
        continue;
      }

      items.add(new ChatMessageItem(message));
      items.add(new DateItem(message.getCreatedAt()));

      if (message.getProfileBot() != null) {
        items.add(new ProfileBotMessageItem(message));
        continue;
      }

      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        SendingMessageItem item = SendingMessageItem.create(message.getRequestId());
        if (item != null) {
          items.remove(item);
        }
      }
    }

    items.endBatchedUpdates();
  }

  @Override
  public void addMessageItem(MessageItem item) {
    items.add(item);
  }

  @Override
  public void addMessageItems(Collection<MessageItem> items) {
    this.items.addAll(items);
  }

  @Override
  public void removeMessageItem(@Nullable MessageItem item) {
    if (item == null) {
      return;
    }
    items.remove(item);
  }

  @Override
  public void addOrUpdateMessageItem(int position, MessageItem item) {
    if (position >= 0) {
      items.updateItemAt(position, item);
    } else {
      addMessageItem(item);
    }
  }

  @Nullable
  @Override
  public MessageItem getItem(int index) {
    if (index < 0 || index >= items.size()) {
      return null;
    }
    return items.get(index);
  }

  @Nullable
  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    LayoutInflater i = LayoutInflater.from(parent.getContext());
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case SENDING:
        return SendingMessageHolder.newInstance(parent, type, onSendingMessageClickListener);

      case DATE:
        return new DateDividerHolder(
            i.inflate(R.layout.ch_plugin_item_message_date, parent, false));

      case NEW_MESSAGE_DIVIDER:
        return new NewMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_unread_divider, parent, false));

      case LOG:
        return new LogMessageHolder(i.inflate(R.layout.ch_plugin_item_message_log, parent, false));

      case TYPING:
        return TypingHolder.newInstance(parent);

      case GUEST_TEXT_MESSAGE:
      case MANAGER_TEXT_MESSAGE:
      case BOT_TEXT_MESSAGE:
        return TextMessageHolder.newInstance(parent, type, onTextMessageClickListener);

      case GUEST_IMAGE_MESSAGE:
      case MANAGER_IMAGE_MESSAGE:
      case BOT_IMAGE_MESSAGE:
        return ImageMessageHolder.newInstance(parent, type, onImageMessageClickListener);

      case GUEST_FILE_MESSAGE:
      case MANAGER_FILE_MESSAGE:
      case BOT_FILE_MESSAGE:
        return FileMessageHolder.newInstance(parent, type, onFileMessageClickListener);

      case GUEST_WEB_MESSAGE:
      case MANAGER_WEB_MESSAGE:
      case BOT_WEB_MESSAGE:
        return WebMessageHolder.newInstance(parent, type, onWebMessageClickListener);

      case PROFILE_BOT_MESSAGE:
        return ProfileBotMessageHolder.newInstance(parent, onProfileUpdateListener);

      case FORM_MESSAGE:
        return FormMessageHolder.newInstance(
            parent,
            type,
            onFormMessageClickListener);

      case WELCOME_MESSAGE:
        return WelcomeMessageHolder.newInstance(parent, onTextMessageClickListener);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case SENDING:
        SendingMessageItem sendingMessageItem = (SendingMessageItem) item;
        SendingMessageHolder sendingHolder = (SendingMessageHolder) holder;
        sendingHolder.bind(
            sendingMessageItem,
            sendingMessageItem.isConnected(getItem(position - 1)));
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.set((DateItem) item);
        break;

      case LOG:
        LogMessageHolder logMessageHolder = (LogMessageHolder) holder;
        logMessageHolder.bind((LogMessageItem) item);
        break;

      case TYPING:
        ((TypingHolder) holder).bind((TypingItem) item);
        break;

      case GUEST_TEXT_MESSAGE:
      case MANAGER_TEXT_MESSAGE:
      case BOT_TEXT_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;
        TextMessageHolder textMessageHolder = (TextMessageHolder) holder;
        textMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case GUEST_FILE_MESSAGE:
      case MANAGER_FILE_MESSAGE:
      case BOT_FILE_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;
        FileMessageHolder fileMessageHolder = (FileMessageHolder) holder;
        fileMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case GUEST_IMAGE_MESSAGE:
      case MANAGER_IMAGE_MESSAGE:
      case BOT_IMAGE_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;
        ImageMessageHolder imageMessageHolder = (ImageMessageHolder) holder;
        imageMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case GUEST_WEB_MESSAGE:
      case MANAGER_WEB_MESSAGE:
      case BOT_WEB_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;
        WebMessageHolder webMessageHolder = (WebMessageHolder) holder;
        webMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case PROFILE_BOT_MESSAGE:
        ProfileBotMessageItem profileItem = (ProfileBotMessageItem) item;
        ProfileBotMessageHolder profileBotMessageHolder = (ProfileBotMessageHolder) holder;

        profileBotMessageHolder.bind(profileItem, getChatEditTextFocus());
        break;

      case FORM_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;
        FormMessageHolder formMessageHolder = (FormMessageHolder) holder;
        formMessageHolder.bind(
            chatMessageItem,
            position == getItemCount() - 2,
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case WELCOME_MESSAGE:
        WelcomeMessageItem welcomeMessageItem = (WelcomeMessageItem) item;
        WelcomeMessageHolder welcomeMessageHolder = (WelcomeMessageHolder) holder;

        welcomeMessageHolder.bind(welcomeMessageItem);
        break;
    }
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public int getIndex(MessageItem item) {
    return items.indexOf(item);
  }

  @Override
  public void setImageFilesToStorage() {
    PhotoAlbumStorage instance = PhotoAlbumStorage.getInstance();

    for (int i = 0; i < items.size(); i++) {
      MessageItem messageItem = items.get(i);

      if (messageItem.getType() == MessageType.MANAGER_IMAGE_MESSAGE ||
          messageItem.getType() == MessageType.GUEST_IMAGE_MESSAGE ||
          messageItem.getType() == MessageType.BOT_IMAGE_MESSAGE) {
        ChatMessageItem chatMessageItem = (ChatMessageItem) messageItem;
        Message message = chatMessageItem.getMessage();
        File file = message.getFile();

        if (file != null && file.isImage()) {
          if (!TextUtils.isEmpty(file.getUrl())
              && !TextUtils.isEmpty(file.getFilename())
              && file.getSize() > 0) {
            instance.addImageFile(file);
          }
        }
      }
    }
  }

  @Nullable
  @Override
  public String getLastMessageItemId() {
    if (items.size() > 1) {
      MessageItem messageItem = items.get(items.size() - 2);
      return String.format("%s:%s", messageItem.getPrimaryKey(), messageItem.getSecondaryKey());
    }
    return null;
  }

  @Nullable
  @Override
  public WelcomeMessageItem getWelcomeMessage() {
    return welcomeMessageItem;
  }

  @Override
  public void setWelcomeMessage(String message, Long timestamp) {
    if (this.welcomeMessageItem != null) {
      items.remove(this.welcomeMessageItem);
    }
    this.welcomeMessageItem = new WelcomeMessageItem(message, timestamp);
    addMessageItem(this.welcomeMessageItem);
  }

  public void setChatEditTextFocus(boolean hasFocus) {
    hasChatEditTextFocus = hasFocus;
  }

  private boolean getChatEditTextFocus() {
    return hasChatEditTextFocus;
  }

  @Nullable
  public MessageItem get(int position) {
    if (position < 0 || position >= items.size()) {
      return null;
    }
    return items.get(position);
  }
}
