package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.event.BadgeBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.PushBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.etc.PushEvent;
import com.zoyi.channel.plugin.android.model.wrapper.ChannelWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.GuestWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

public class ChannelIOManager {

  @Nullable
  private static ChannelIOManager channelIOManager;

  private Application application;

  private int activityRunningCount = 0;

  @Nullable
  private Subscription socketSubscription;

  private List<ChannelPluginListener> listeners;

  static void create(Application application) {
    if (channelIOManager == null) {
      channelIOManager = new ChannelIOManager(application);
    }
  }

  private ChannelIOManager(Application application) {
    this.application = application;
    this.listeners = new ArrayList<>();
  }

  static void bootPlugin(PluginWrapper pluginWrapper) {
    if (channelIOManager != null) {
      PrefSupervisor.setVeilId(channelIOManager.application, pluginWrapper.getVeilId());
      channelIOManager.startReceiveSocket();
    }
  }

  static void clear() {
    if (channelIOManager != null) {
      channelIOManager.stopReceiveSocket();
    }
  }

  static void fetchMe(@Nullable final OnGuestUpdatedListener listener) {
    if (channelIOManager != null) {
      ChannelIO.getApi().getMe()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<GuestWrapper>() {
            @Override
            public void onNext(GuestWrapper repo) {
              if (repo != null) {
                repo.update();

                if (listener != null) {
                  listener.onGuestUpdated();
                }
              }
            }
          });
    }
  }

  private void startReceiveSocket() {
    if (socketSubscription == null || socketSubscription.isUnsubscribed()) {
      socketSubscription = RxBus.observable()
          .subscribe(new Action1<Object>() {
            @Override
            public void call(Object o) {
              if (o instanceof CommandBus) {
                CommandBus commandBus = (CommandBus) o;

                switch (commandBus.getCommand()) {
                  case READY:
                    fetchMe(null);
                    break;
                }
              }

              if (o instanceof BadgeBus) {
                for (ChannelPluginListener listener : listeners) {
                  if (listener != null) {
                    listener.onChangeBadge(((BadgeBus) o).getCount());
                  }
                }
              }

              if (o instanceof PushBus) {
                PushBus pushBus = (PushBus) o;
                for (ChannelPluginListener listener : listeners) {
                  if (listener != null) {
                    listener.onReceivePush(new PushEvent(pushBus));
                  }
                }
              }
            }
          });
    }
  }

  private void stopReceiveSocket() {
    if (socketSubscription != null && !socketSubscription.isUnsubscribed()) {
      socketSubscription.unsubscribe();
    }
  }

  public static void increaseActivityRunningCount() {
    if (channelIOManager != null) {
      channelIOManager.activityRunningCount++;
    }
  }

  public static void decreaseActivityRunningCount() {
    if (channelIOManager != null) {
      channelIOManager.activityRunningCount--;
    }
  }

  public static boolean isMainRunning() {
    return channelIOManager != null && channelIOManager.activityRunningCount > 0;
  }

  @NonNull
  static List<ChannelPluginListener> getChannelPluginListeners() {
    if (channelIOManager == null) {
      return new ArrayList<>();
    }
    return channelIOManager.listeners;
  }

  static void addChannelPluginListener(ChannelPluginListener listener) {
    if (channelIOManager != null && channelIOManager.listeners != null && listener != null) {
      channelIOManager.listeners.add(listener);
    }
  }

  static void removeChannelPluginListener(ChannelPluginListener listener) {
    if (channelIOManager != null && channelIOManager.listeners != null && listener != null) {
      channelIOManager.listeners.remove(listener);
    }
  }
}
