package com.zoyi.channel.plugin.android.network;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import com.zoyi.channel.plugin.android.model.error.ExceptionResponse;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ParseUtils;
import com.zoyi.okhttp3.ResponseBody;
import com.zoyi.retrofit2.HttpException;
import com.zoyi.retrofit2.Response;

import java.io.IOException;

/**
 * Created by mika on 2018. 7. 13..
 */

public class RetrofitException extends RuntimeException {

  @NonNull
  private HttpStatus httpStatus;
  @Nullable
  private String exceptionMessage;

  public RetrofitException(@NonNull HttpStatus httpStatus, @Nullable String exceptionMessage) {
    this.httpStatus = httpStatus;
    this.exceptionMessage = exceptionMessage;
  }

  @Override
  public String getMessage() {
    if (exceptionMessage != null) {
      return exceptionMessage;
    }
    return httpStatus.message();
  }

  public boolean isConnectionError() {
    return HttpStatus.CONNECTION_TIMED_OUT.equals(httpStatus);
  }

  public boolean isUnknownError() {
    return HttpStatus.UNKNOWN.equals(httpStatus);
  }

  public boolean isHttpError() {
    return !isConnectionError() && !isUnknownError();
  }

  public static RetrofitException createNetworkException() {
    return new RetrofitException(HttpStatus.CONNECTION_TIMED_OUT, null);
  }

  public static RetrofitException createUnknownException(@Nullable Throwable throwable) {
    if (throwable != null && throwable.getMessage() != null) {
      return new RetrofitException(HttpStatus.UNKNOWN, throwable.getMessage());
    }
    return new RetrofitException(HttpStatus.UNKNOWN, null);
  }

  public static RetrofitException createException(@Nullable Throwable throwable) {
    if (throwable != null) {
      if (throwable instanceof HttpException) {
        try {
          HttpException httpException = (HttpException) throwable;
          Response response = httpException.response();
          HttpStatus httpStatus = HttpStatus.valueOf(response.code());
          ResponseBody errorBody = response.errorBody();

          if (errorBody != null) {
            ExceptionResponse exceptionResponse = ParseUtils.parseJson(errorBody.string(), ExceptionResponse.class);
            if (exceptionResponse != null) {
              return new RetrofitException(httpStatus, exceptionResponse.getMessage());
            }
          }
          return new RetrofitException(httpStatus, null);
        } catch (Exception ex) {
          L.e(ex.getMessage());
        }
      }
      if (throwable instanceof IOException) {
        return createNetworkException();
      }
    }
    return createUnknownException(throwable);
  }
}
