package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.support.annotation.AnimRes;
import android.support.annotation.Nullable;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.activity.chat.ChatActivity;
import com.zoyi.channel.plugin.android.activity.settings.SettingsActivity;
import com.zoyi.channel.plugin.android.activity.userchat_list.contract.UserChatListContract;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.StoreType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.enumerate.UpdateType;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatsWrapper;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.ChannelSelector;
import com.zoyi.channel.plugin.android.selector.ColorSelector;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.store.UserChatStore;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.view.external.fab.FloatingActionButton;
import com.zoyi.channel.plugin.android.view.handler.AlertDialogListener;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.BigBar.MenuPosition;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;
import com.zoyi.channel.plugin.android.view.layout.WaterMarkLayout;

import static com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;

/**
 * Created by mika on 2016. 12. 5..
 */
public class UserChatListActivity extends BaseActivity implements View.OnClickListener, UserChatListContract.View {

  private View rootView;
  private View emptyLayout;
  private View reconnectLayout;

  private RecyclerView chatList;
  private WaterMarkLayout waterMarkLayout;
  private FloatingActionButton fab;
  private LinearLayoutManager layoutManager;

  private UserChatListAdapter adapter;
  private UserChatListContract.Presenter presenter;

  @Nullable
  private String redirectChatId;
  protected boolean animateOnInit = true;

  @Override
  protected void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    init(R.layout.ch_plugin_activity_user_chat_list);

    // get redirecting chat id, it will be turn on after user chat list is loaded

    if (getIntent() != null) {
      redirectChatId = getIntent().getStringExtra(Const.EXTRA_CHAT_ID);
      animateOnInit = getIntent().getBooleanExtra(Const.EXTRA_REDIRECT_ANIMATED, true);
    }

    // view initialize

    rootView = findViewById(R.id.root_view);
    emptyLayout = findViewById(R.id.layout_empty);
    reconnectLayout = findViewById(R.id.layout_reconnect);
    fab = findViewById(R.id.fab);
    waterMarkLayout = findViewById(R.id.chatlist_layout_water_mark);
    chatList = findViewById(R.id.recycler_view);

    // floating action view

    fab.setOnClickListener(this);

    // big bar setting

    if (bigBar != null) {
      bigBar.withActivity(this)
          .addMenu(ActionType.SETTINGS, MenuPosition.LEFT)
          .addMenu(ActionType.REFRESH, MenuPosition.RIGHT, MenuState.HIDDEN)
          .addMenu(ActionType.EXIT, MenuPosition.RIGHT)
          .build();
    }

    // initialize components

    adapter = new UserChatListAdapter();

    presenter = new UserChatListPresenter(this);
    presenter.setView(this);
    presenter.setAdapterView(adapter);
    presenter.setAdapterModel(adapter);

    layoutManager = new LinearLayoutManager(this);
    chatList.setLayoutManager(layoutManager);
    chatList.setAdapter(adapter);
    chatList.setItemAnimator(null);
    chatList.getRecycledViewPool().setMaxRecycledViews(0, 40);

    // Set view style
    setStyle();
    setWaterMarkLayout();

    // Socket error handling
    if (SocketManager.isError()) {
      setErrorView(true);
    }

    // Set user chat filter
    Store.getInstance(UserChatStore.class).setShowClosed(PrefSupervisor.isShownClosedChat(this));

    presenter.refresh();
  }

  private void setStyle() {
    Channel channel = ChannelSelector.getCurrentChannel();
    Plugin plugin = PluginSelector.getPlugin();

    if (channel == null || plugin == null) {
      finish();
    } else {
      if (bigBar != null) {
        bigBar.setTitle(channel.getName());
      }

      setFabTheme();
      fab.attachToRecyclerView(chatList);
    }
  }

  private void setWaterMarkLayout() {
    final Channel channel = ChannelSelector.getCurrentChannel();

    if (channel != null && channel.isTrial()) {
      waterMarkLayout.setVisibility(View.VISIBLE);

      waterMarkLayout.setOnClickListener(new View.OnClickListener() {
        @Override
        public void onClick(View v) {
          Executor.goWeb(getBaseContext(),
              String.format(
                  Const.UTM_URL,
                  PrefSupervisor.getPluginLanguage(UserChatListActivity.this),
                  "plugin_watermark",
                  channel.getName(),
                  channel.getId()));
        }
      });

      chatList.addOnScrollListener(new InfiniteScrollListener(
          layoutManager,
          InfiniteScrollListener.BOTTOM,
          chatList) {

        @Override
        public void scrolledInList() {
          waterMarkLayout.setVisibility(View.GONE);
        }

        @Override
        public void scrollAttachedToBottom() {
          waterMarkLayout.setVisibility(View.VISIBLE);
        }

        @Override
        public void refresh() {
          // not yet.
        }
      });
    } else {
      waterMarkLayout.setVisibility(View.GONE);
      chatList.setPadding(0, 0, 0, 0);
    }
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);

    switch (requestCode) {
      case Const.REQUEST_PLUGIIN_SETTINGS:
        if (resultCode == Const.RESULT_CHAT_OPTION_CHANGED) {
          presenter.refresh();
        }
        presenter.redrawList();
        break;

      case Const.REQUEST_CHAT:
        switch (resultCode) {
          case Const.RESULT_CHAT_OPEN_NEW_CHAT:
            startChat(null);
            break;

          case Const.RESULT_CHAT_OPTION_CHANGED:
            presenter.refresh();
            break;

          case Const.RESULT_CHAT_GO_TO_MAIN:
            finish();
            break;
        }
        break;
    }
  }

  @Override
  protected void onStart() {
    ChannelPushManager.sendTokenToChannelIO(ChannelIO.getAppContext());
    super.onStart();
  }

  private void setFabTheme() {
    fab.setColorNormal(ColorSelector.getBackgroundColor());
    fab.setColorPressed(Utils.addBlackMask(ColorSelector.getTextColor()));
    fab.setColorRipple(Color.WHITE);
    fab.setImageResource(
        ColorSelector.getTextColor() == Color.BLACK
            ? R.drawable.ch_plugin_ic_add_black_24dp
            : R.drawable.ch_plugin_ic_add_white_24dp);
  }

  @Override
  public void onClick(View view) {
    int id = view.getId();

    if (id == R.id.fab) {
      startChat(null);
    }
  }

  @Override
  public void optionClicked(ActionType actionType) {
    switch (actionType) {
      case REFRESH:
        if (bigBar != null) {
          bigBar.setMenuState(ActionType.REFRESH, MenuState.HIDDEN);
        }
        presenter.refresh();
        break;

      case SETTINGS:
        IntentUtils.setNextActivity(this, SettingsActivity.class)
            .setTransition(Transition.SLIDE_FROM_BOTTOM)
            .startActivityForResult(Const.REQUEST_PLUGIIN_SETTINGS);
        break;
    }
    super.optionClicked(actionType);
  }

  @Override
  public void userChatsFetched(UserChatsWrapper userChatsWrapper, boolean init) {
    if (redirectChatId != null) {
      startChat(redirectChatId, true, animateOnInit);
      redirectChatId = null;
    } else {
      if (adapter.getItemCount() <= 1 && init) {
        if (adapter.getItemCount() == 0) {
          startChat(null, true, animateOnInit);
        } else {
          String chatId = adapter.getChatId(0);
          startChat(chatId, true, animateOnInit);
        }
      }
    }

    setViewVisibility();
  }

  @Override
  public void startChat(@Nullable String chatId) {
    startChat(chatId, false, true);
  }

  private void startChat(@Nullable String chatId, boolean redirect, boolean animate) {
    Transition transition;

    if (!animate) {
      // set none when animate is false
      transition = Transition.NONE;
    } else if (redirect) {
      // when launch with redirect chat id, always show from bottom
      transition = Transition.SLIDE_FROM_BOTTOM;
    } else {
      // else transition is normal
      transition = Transition.SLIDE_FROM_RIGHT;
    }

    presenter.fetchPreloadData(chatId, transition);
  }

  @Override
  public void setErrorView(boolean show) {
    if (bigBar != null) {
      bigBar.setMenuState(ActionType.REFRESH, show ? MenuState.VISIBLE : MenuState.HIDDEN);
    }
    Views.setVisibility(reconnectLayout, show);
  }

  @Override
  public void showDeleteUserChatDialog(final String chatId) {
    UIUtils.showAlertDialog(
        this,
        ResUtils.getString(this, "ch.chat.delete_description"),
        true,
        new AlertDialogListener() {
          @Override
          public void onPositive() {
            presenter.removeUserChat(chatId);
          }
        });
  }

  @Override
  public void completeRemoveChat() {
    hideProgress();
    setViewVisibility();
  }

  @Override
  public void successFetchPreloadData(String chatId, Transition transition) {
    IntentUtils.setNextActivity(this, ChatActivity.class)
        .putExtra(Const.EXTRA_CHAT_ID, chatId)
        .setTransition(transition)
        .startActivityForResult(Const.REQUEST_CHAT);
  }

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    presenter.receiveCommand(command, object);
  }

  @Override
  public void receiveData(Entity entity, boolean upsert) {
    presenter.receiveData(entity, upsert);

    setViewVisibility();
  }

  @Override
  public void receiveStoreEvent(StoreType storeType, UpdateType updateType, @Nullable Entity entity) {
    if (StoreType.PLUGIN == storeType) {
      setFabTheme();
    }
    presenter.receiveStoreEvent(storeType, updateType, entity);
  }

  private void setViewVisibility() {
    emptyLayout.setVisibility(UIUtils.getVisible(!hasUserChat(), true));
    chatList.setVisibility(UIUtils.getVisible(hasUserChat(), true));
  }

  private boolean hasUserChat() {
    return adapter.getItemCount() > 0;
  }

  @Override
  @AnimRes
  protected int getExistAnimOfFinish() {
    return R.anim.ch_plugin_slide_out_bottom;
  }
}
