package com.zoyi.channel.plugin.android.network;

import com.zoyi.channel.plugin.android.model.Error;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.com.google.gson.Gson;
import com.zoyi.com.google.gson.reflect.TypeToken;
import com.zoyi.retrofit2.Response;
import com.zoyi.retrofit2.adapter.rxjava.HttpException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2017. 1. 9..
 */
public class RetrofitException extends RuntimeException {
  private static RetrofitException networkError(IOException exception) {
    return new RetrofitException(
        Kind.NETWORK,
        new ArrayList<>(Collections.singleton(new Error("Network Error"))));
  }

  private static RetrofitException unexpectedError(Throwable exception) {
    return new RetrofitException(
        Kind.UNEXPECTED,
        new ArrayList<>(Collections.singleton(new Error(exception.getMessage()))));
  }

  public static RetrofitException create(Throwable throwable) {
    if (throwable instanceof HttpException) {
      HttpException httpException = (HttpException) throwable;
      Response response = httpException.response();

      Gson gson = new Gson();

      try {
        JSONObject jsonObject = new JSONObject(response.errorBody().string());

        if (jsonObject.has("error")) {
          return new RetrofitException(
              Kind.HTTP,
              new ArrayList<>(Collections.singleton(gson.fromJson(jsonObject.getString("error"), Error.class))));
        } else if (jsonObject.has("errors")) {
          List<Error> errors = gson.fromJson(
              jsonObject.getString("errors"),
              new TypeToken<List<Error>>() {}.getType());
          return new RetrofitException(Kind.HTTP, errors);
        } else {
          return unexpectedError(throwable);
        }
      } catch (Exception e) {
        L.e(e.getMessage());
        return unexpectedError(throwable);
      }
    }

    if (throwable instanceof IOException) {
      return networkError((IOException) throwable);
    }

    return unexpectedError(throwable);
  }

  public enum Kind {
    NETWORK,
    HTTP,
    UNEXPECTED
  }

  private List<Error> errors;
  private final Kind kind;

  private RetrofitException(Kind kind, List<Error> errors) {
    this.kind = kind;
    this.errors = errors;
  }

  public Kind getKind() {
    return kind;
  }

  @Override
  public String getMessage() {
    List<String> list = new ArrayList<>();
    for (Error error : errors) {
      list.add(error.getMessage());
    }
    return StringUtils.join(list, "\n");
  }
}

