package com.zoyi.channel.plugin.android.activity.chat;

import android.support.v4.util.LongSparseArray;

import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.model.MessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.wrapper.MessageWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.retrofit2.okhttp3.MediaType;
import com.zoyi.retrofit2.okhttp3.MultipartBody;
import com.zoyi.retrofit2.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

/**
 * Created by mika on 2017. 3. 2..
 */
public class ChatManager {
  private static ChatManager chatManager;

  public static ChatManager get() {
    if (chatManager == null) {
      synchronized (ChatManager.class) {
        if (chatManager == null) {
          chatManager = new ChatManager();
        }
      }
    }
    return chatManager;
  }

  private boolean sending;
  private boolean released;

  private Queue<SendingMessageItem> sendingItems;
  private LongSparseArray<SendingMessageItem> failedItems;

  public ChatManager() {
    released = false;
    sending = false;
    sendingItems = new LinkedList<>();
    failedItems = new LongSparseArray<>();
  }

  public void removeFailedItem(Long createdAt) {
    if (createdAt != null) {
      failedItems.remove(createdAt);
    }
  }

  public void send(String chatId, List<SendingMessageItem> items, OnMessageSendListener listener) {
    if (!released) {
      for (SendingMessageItem item : items) {
        item.setChatId(chatId);
        send(item, listener);
      }
    }
  }

  public void send(List<SendingMessageItem> items, OnMessageSendListener listener) {
    if (!released) {
      for (SendingMessageItem item : items) {
        send(item, listener);
      }
    }
  }

  public void send(SendingMessageItem item, OnMessageSendListener listener) {
    if (!released && item != null && item.getCreatedAt() != null) {
      removeFailedItem(item.getCreatedAt());
      item.setSending(true);
      sendingItems.add(item);
      send(listener);
    }
  }

  private void send(OnMessageSendListener listener) {
    if (!sending && sendingItems.size() > 0) {
      sending = true;
      final SendingMessageItem item = sendingItems.poll();
      if (item.isImage()) {
        sendFileMessage(item, listener);
      } else {
        sendTextMessage(item, listener);
      }
    }
  }

  private void sendTextMessage(final SendingMessageItem item, final OnMessageSendListener listener) {
    RequestBody body = RequestUtils.form()
        .set("message", item.getMessage())
        .set("requestId", item.getRequestId())
        .create();

    ChannelPlugin.getApi().sendMessage(item.getChatId(), body)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessageWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            failed(error, item, listener);
          }

          @Override
          public void onNext(MessageWrapper wrapper) {
            successed(item, wrapper.getMessage(), listener);
          }
        });
  }

  private void sendFileMessage(final SendingMessageItem item, final OnMessageSendListener listener) {
    MultipartBody.Part body = RequestUtils.makeMultipart(item.getString());
    RequestBody timestamp = RequestBody.create(MediaType.parse("text/plane"), item.getRequestId());

    if (body == null) {
      failed(new Exception("Not supported type"), item, listener);
      return;
    }

    ChannelPlugin.getApi().uploadFile(item.getChatId(), body, timestamp)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessageWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            failed(error, item, listener);
          }

          @Override
          public void onNext(MessageWrapper wrapper) {
            successed(item, wrapper.getMessage(), listener);
          }
        });
  }

  private void successed(SendingMessageItem item, Message message, OnMessageSendListener listener) {
    if (!released) {
      if (listener != null) {
        listener.sendSuccessed(item, message);
      }
      sending = false;
      send(listener);
    }
  }

  private void failed(Exception ex, SendingMessageItem item, OnMessageSendListener listener) {
    if (ex != null) {
      L.e(ex.getMessage());
    }

    if (!released) {
      item.setSending(false);

      failedItems.put(item.getCreatedAt(), item);

      if (listener != null) {
        listener.sendFailed(item);
      }
      sending = false;
      send(listener);
    }
  }

  public List<MessageItem> getFailedItems(String chatId) {
    List<MessageItem> items = new ArrayList<>();

    for (int i = 0; i < failedItems.size(); i++) {
      SendingMessageItem item = failedItems.valueAt(i);

      if (CompareUtils.isSame(chatId, item.getChatId())) {
        items.add(item);
      }
    }
    return items;
  }

  public static void release() {
    if (chatManager != null) {
      chatManager.released = true;
      chatManager.sendingItems.clear();
      chatManager.failedItems.clear();

      chatManager = null;
    }
  }
}
