package com.zoyi.channel.plugin.android.presenter.chat;

import android.app.ProgressDialog;
import android.content.Context;
import android.graphics.Color;
import android.os.Handler;
import android.support.annotation.ColorRes;
import android.text.TextUtils;
import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.activity.chat.ChatDataDictionary;
import com.zoyi.channel.plugin.android.activity.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.CountryDictionary;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.ActionType;
import com.zoyi.channel.plugin.android.activity.chat.type.UserInfoType;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.chat.ChatState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.MessagesWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserVeilWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.rx.Observable;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Created by mika on 2017. 2. 27..
 */
public class ChatPresenter implements
    ChatContract.Presenter,
    OnActionMessageClickListener,
    OnSendingMessageClickListener,
    OnMessageClickListener,
    OnUserInfoListener,
    OnMessageSendListener {

  private ChatContract.View view;
  private ChatAdapterContract.View adapterView;
  private ChatAdapterContract.Model adapterModel;

  private Context context;
  private ProgressDialog dialog;

  private UserChat userChat;
  private Session session;

  private ChatDataDictionary dictionary;
  private CountryDictionary countryDictionary;

  private String chatId;
  private int backColor = Color.BLACK;
  private int textColor = Color.WHITE;

  private ChatState state = ChatState.IDLE;

  private String backwardId = Const.MESSAGE_ID_MAX;
  private String forwardId = Const.MESSAGE_ID_MIN;
  private String forwardTempId = Const.MESSAGE_ID_MIN;
  private Long forwardTimestamp = Long.MIN_VALUE;

  private int fetchCounter = 0;
  private boolean welcomeMessageExists = false;
  private boolean backwardLoading = false;
  private boolean newMessageDismiss = false;
  private boolean firstMessage = true;
  private boolean isUserInfoFirst = true;

  private Long lastReadAt;

  private UserInfoItem userInfoItem;

  private List<SendingMessageItem> tempQueue;

  public ChatPresenter(Context context) {
    this.context = context;

    TimeUtils.refreshOffset();

    dictionary = new ChatDataDictionary();
    countryDictionary = new CountryDictionary(context);
    tempQueue = new ArrayList<>();

    ChatManager.get().setOnMessageSendListener(this);
  }

  @Override
  public void showProgress() {
    if (dialog == null || !dialog.isShowing()) {
      dialog = ProgressHelper.show(context, ResUtils.getString(context, "ch.loading_information"));
    }
  }

  @Override
  public void hideProgress() {
    if (dialog != null && dialog.isShowing()) {
      dialog.dismiss();
    }
  }

  @Override
  public void setView(ChatContract.View view) {
    this.view = view;
  }

  @Override
  public String getChatId() {
    return chatId;
  }

  @Override
  public void setChatId(String chatId) {
    this.chatId = chatId;
  }

  @Override
  public int getBackColor() {
    return backColor;
  }

  @Override
  public int getTextColor() {
    return textColor;
  }

  @Override
  public void setColor(@ColorRes int backColor, @ColorRes int textColor) {
    this.backColor = backColor;
    this.textColor = textColor;
  }

  @Override
  public void setAdapterView(ChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
    this.adapterView.setOnMessageClickListener(this);
    this.adapterView.setOnActionMessageClickListener(this);
    this.adapterView.setOnSendingMessageClickListener(this);
    this.adapterView.setOnUserInfoListener(this);
  }

  @Override
  public void setAdapterModel(ChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void init() {
    if (chatId == null) {
      addWelcomeMessage();
    } else {
      adapterModel.addMessageItems(ChatManager.get().getFailedItems(chatId));

      fetchUserChat();
    }
  }

  private void addWorkingTimeMessage() {
    if (!welcomeMessageExists) {
      if (!ChannelStore.isWorking()) {
        String message = ChannelStore.getScript("out_of_work", "ch.scripts.out_of_work.default");
        if (message != null) {
          welcomeMessageExists = true;
          adapterModel.addMessageItem(
              new ActionMessageItem(
                  ActionType.REQUEST_WORKING_TIME,
                  message,
                  forwardTimestamp,
                  1)
          );
        }
      }
    }
  }

  private void addWelcomeMessage() {
    if (!welcomeMessageExists) {
      if (!ChannelStore.isWorking()) {
        addWorkingTimeMessage();
      } else {
        String message;
        User user = ChannelStore.getUser();
        if (user != null && user.getName() != null) {
          message = ChannelStore.getScript("welcome_user", "ch.scripts.welcome_user.default");
          if (message != null) {
            message = message.replace("${user}", user.getName());
          }
        } else {
          message = ChannelStore.getScript("welcome_veil", "ch.scripts.welcome_veil.default");
        }

        if (message != null) {
          welcomeMessageExists = true;
          adapterModel.addMessageItem(
              new ActionMessageItem(
                  null,
                  message,
                  forwardTimestamp,
                  1)
          );
        }
      }
    }
  }

  @Override
  public void fetchUserChat() {
    if (userChat == null) {
      showProgress();
    }
    setState(ChatState.USER_CHAT_LOADING, false);
    ChannelPlugin.getApi().getUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            hideProgress();
            handleUserChatFetchError(error);
          }

          @Override
          public void onNext(UserChatWrapper userChatWrapper) {
            setUserChat(userChatWrapper, false);
          }
        });
  }

  private void createUserChat() {
    showProgress();

    setState(ChatState.USER_CHAT_LOADING, false);

    if (ChannelStore.getPluginId() != null) {
      ChannelPlugin.getApi().createUserChat(ChannelStore.getPluginId())
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              hideProgress();
              handleUserChatFetchError(error);
            }

            @Override
            public void onNext(UserChatWrapper userChatWrapper) {
              setUserChat(userChatWrapper, true);
            }
          });
    } else {
      hideProgress();
    }
  }

  private void setUserChat(UserChatWrapper wrapper, boolean create) {
    hideProgress();

    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      if (wrapper == null) {
        handleUserChatFetchError(new Exception("UserChatWrapper cannot be null"));
      } else {
        view.onLoadUserChat();
        view.setReconnectVisibility(false);

        userChat = wrapper.getUserChat();
        chatId = wrapper.getUserChat().getId();
        session = wrapper.getSession();
        dictionary.add(wrapper.getManagers());

        if (lastReadAt == null) {
          lastReadAt = create ? Long.MAX_VALUE : session.getLastReadAt();
        }

        if (tempQueue.size() > 0) {
          ChatManager.get().send(chatId, tempQueue);
          tempQueue.clear();
        }

        joinChat();
      }
    }
  }

  private void handleUserChatFetchError(Throwable throwable) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      setState(ChatState.USER_CHAT_NOT_LOADED, true);
    }
  }

  private void joinChat() {
    if (SocketManager.isReady()) {
      setState(ChatState.CHAT_JOINING, false);
      SocketManager.joinChat(chatId);
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  private boolean isMessageExists() {
    return CompareUtils.compare(backwardId, forwardId) < 0;
  }

  private boolean isAvailableRequest(int count) {
    return isStateEquals(ChatState.MESSAGES_LOADING) && fetchCounter == count;
  }

  @Override
  public void fetchMessages() {
    if (SocketManager.isReady()) {
      fetchCounter++;
      setState(ChatState.MESSAGES_LOADING, false);
      if (!isMessageExists()) {
        fetchInitMessages(fetchCounter);
      } else {
        fetchForwardMessages(forwardId, fetchCounter);
      }
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  @Override
  public void fetchInitMessages(final int counter) {
    ChannelPlugin.getApi()
        .getMessages(
            chatId,
            Const.MESSAGE_ID_MAX,
            Const.MESSAGE_FETCH_LIMIT,
            Const.DESC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            if (isAvailableRequest(counter)) {
              handleMessagesFetchError(error, counter);
            }
          }

          @Override
          public void onNext(MessagesWrapper wrapper) {
            if (isAvailableRequest(counter)) {
              if (wrapper == null) {
                handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
              } else {
                backwardId = wrapper.getNext();
                dictionary.add(wrapper.getManagers());
                dictionary.add(wrapper.getBots());

                addMessages(wrapper.getMessages());
                calculateForwardId(wrapper.getMessages());
                calculateForwardId();

                if (lastReadAt >= forwardTimestamp) {
                  lastReadAt = Long.MAX_VALUE;
                }

                addNewMessageItem(wrapper.getMessages());

                addWorkingTimeMessage();

                setState(ChatState.CHAT_READY, false);
                readAll();
              }
            }
          }
        });
  }

  @Override
  public void fetchBackwardMessages() {
    if (backwardLoading || !isMessageExists()) {
      return;
    }
    backwardLoading = true;

    ChannelPlugin.getApi()
        .getMessages(
            chatId,
            backwardId,
            Const.MESSAGE_FETCH_LIMIT,
            Const.DESC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            backwardLoading = false;

            handleMessagesFetchError(error, fetchCounter);
          }

          @Override
          public void onNext(MessagesWrapper wrapper) {
            if (wrapper == null) {
              handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), fetchCounter);
            } else {
              backwardId = wrapper.getNext();

              dictionary.add(wrapper.getManagers());
              dictionary.add(wrapper.getBots());

              adapterModel.addMessages(wrapper.getMessages(), chatId);

              addNewMessageItem(wrapper.getMessages());
            }
            backwardLoading = false;
          }
        });
  }

  @Override
  public void fetchForwardMessages(String since, final int counter) {
    ChannelPlugin.getApi()
        .getMessages(
            chatId,
            since,
            Const.MESSAGE_FETCH_LIMIT,
            Const.ASC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            if (isAvailableRequest(counter)) {
              handleMessagesFetchError(error, counter);
            }
          }

          @Override
          public void onNext(MessagesWrapper wrapper) {
            if (isAvailableRequest(counter)) {
              if (wrapper == null) {
                handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
              } else {
                dictionary.add(wrapper.getManagers());
                dictionary.add(wrapper.getBots());

                addMessages(wrapper.getMessages());
                calculateForwardId(wrapper.getMessages());

                if (wrapper.getNext() == null) {
                  calculateForwardId();
                  setState(ChatState.CHAT_READY, false);
                  readAll();
                } else {
                  fetchForwardMessages(forwardId, counter);
                }
              }
            }
          }
        });
  }

  private void handleMessagesFetchError(Throwable throwable, int counter) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.MESSAGES_LOADING) && fetchCounter == counter) {
      setState(ChatState.MESSAGES_NOT_LOADED, true);
    }
  }

  @Override
  public void calculateNewToast(int index) {
    if (lastReadAt == null || newMessageDismiss) {
      return;
    }

    MessageItem item = adapterModel.getItem(index);

    if (item != null) {
      if (lastReadAt < item.getFirst()) {
        view.setToastVisibility(true);
      } else {
        hideNewToast();
      }
    }
  }

  @Override
  public void newToastClicked() {
    if (lastReadAt != null) {
      NewMessageItem item = new NewMessageItem(lastReadAt);

      int index = adapterModel.getIndex(item);

      if (index < 0) {
        view.scrollTo(0);
      } else {
        hideNewToast();
        view.scrollTo(index);
      }
    }
  }

  @Override
  public void hideNewToast() {
    newMessageDismiss = true;
    view.setToastVisibility(false);
  }

  private void addNewMessageItem(List<Message> messages) {
    if (lastReadAt == null) {
      return;
    }

    long min = Long.MAX_VALUE;
    long max = Long.MIN_VALUE;

    if (messages != null) {
      for (Message message : messages) {
        if (message.getLog() != null) {
          continue;
        }
        min = Math.min(min, message.getCreatedAt());
        max = Math.max(max, message.getCreatedAt());
      }
    }

    if (min <= lastReadAt && lastReadAt < max) {
      adapterModel.addMessageItem(new NewMessageItem(lastReadAt));
    }
  }

  private void addMessage(Message message) {
    if (CompareUtils.isSame(message.getChatId(), chatId)) {
      adapterModel.addMessage(message, chatId);

      boolean isMine = CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME);

      if (isStateEquals(ChatState.CHAT_READY)) {
        calculateForwardId(message);
        readAll();
      } else {
        forwardTempId = CompareUtils.max(forwardTempId, message.getId());
      }

      forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());

      if (firstMessage && isMine) {
        firstMessage = false;
        checkRequireUserInfo(forwardTimestamp);
      }

      view.scrollToBottom(false);
    }
  }

  private void addMessages(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());
        forwardId = CompareUtils.max(forwardId, message.getId());
      }
    }
    adapterModel.addMessages(messages, chatId);
  }

  private void calculateForwardId(Message message) {
    if (message != null) {
      forwardId = CompareUtils.max(forwardId, message.getId());
    }
  }

  private void calculateForwardId(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        calculateForwardId(message);
      }
    }
  }

  private void calculateForwardId() {
    forwardId = CompareUtils.max(forwardId, forwardTempId);
  }

  @Override
  public void sendTextMessage(String message) {
    if (TextUtils.isEmpty(message)) {
      return;
    }
    SendingMessageItem item = new SendingMessageItem(chatId, message, false);
    adapterModel.addMessageItem(item);

    sendMessages(Collections.singletonList(item));
  }

  @Override
  public void sendImageFiles(ArrayList<String> imagePaths) {
    List<SendingMessageItem> items = new ArrayList<>();
    if (imagePaths != null) {
      for (String p : imagePaths) {
        SendingMessageItem item = new SendingMessageItem(chatId, p, true);
        adapterModel.addMessageItem(item);
        items.add(item);
      }
    }
    sendMessages(items);
  }

  private void sendMessages(List<SendingMessageItem> items) {
    view.scrollToBottom(true);

    if (chatId == null) {
      tempQueue.addAll(items);
      createUserChat();
    } else {
      ChatManager.get().send(items);
    }
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    if (chatId != null && channelModel != null) {
      switch (channelModel.getClass().getSimpleName()) {
        case Message.CLASSNAME:
          addMessage((Message) channelModel);
          break;

        case File.CLASSNAME:
        case WebPage.CLASSNAME:
        case Bot.CLASSNAME:
        case Manager.CLASSNAME:
          dictionary.add(channelModel);
          break;
      }
    }
  }

  private void readAll() {
    ChannelPlugin.getApi().readAll(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>());
  }

  @Override
  public void receiveCommand(Command command, Object object) {
    switch (command) {
      case APP_STARTED:
        refresh();
        break;

      case SOCKET_DISCONNECTED:
        fetchCounter++;
        if (!isStateEquals(ChatState.IDLE)) {
          setState(ChatState.USER_CHAT_NOT_LOADED, true);
        }
        break;

      case SOCKET_ERROR:
        view.setReconnectVisibility(true);
        break;

      case READY:
        view.setReconnectVisibility(false);

        if (isStateEquals(ChatState.WAITING_SOCKET)) {
          joinChat();
        } else {
          refresh();
        }
        break;

      case LEAVED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_READY)) {
          setState(ChatState.NOT_JOINED, false);  // true?
        }
        break;

      case JOINED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_JOINING)) {

          view.setReconnectVisibility(false);

          fetchMessages();
        }
        break;
    }
  }

  @Override
  public void messageClicked(Message message) {
    if (message.getFile() != null) {
      if (message.getFile().getPreviewThumb() == null || !message.getFile().isImage()) {
        view.onFileDownload(message);
      } else {
        adapterModel.setImageFilesToStorage();
        view.onShowPhotoAlbum(message.getFile());
      }
    } else if (message.getWebPage() != null) {
      urlClicked(message.getWebPage().getUrl());
    }
  }

  @Override
  public void urlClicked(String url) {
    view.onUrlClicked(url);
  }

  @Override
  public void actionMessageClicked(ActionMessageItem item) {
    if (item != null && item.getActionType() != null) {
      switch (item.getActionType()) {
        case REQUEST_WORKING_TIME:
          item.setActionType(null);
          adapterModel.addMessageItem(item);

          QuestionMessageItem questionItem = new QuestionMessageItem(
              forwardTimestamp,
              ResUtils.getString(context, "ch.out_of_work.user_answer"),
              item.getOrder() + 1);
          adapterModel.addMessageItem(questionItem);

          view.scrollToBottom(true);

          String workingTime = ChannelStore.getChannel() != null
              ? ChannelStore.getChannel().getWorkingTime(context)
              : "?";
          addMessageDelay(new ActionMessageItem(null, workingTime, null, item.getOrder() + 2));
          break;
      }
    }
  }

  @Override
  public void sendingMessageClicked(SendingMessageItem item) {
    view.sendingMessageClicked(item);
  }

  @Override
  public void resend(SendingMessageItem item) {
    if (item != null) {
      removeFailedItem(item);

      item.refreshCreatedAt();

      adapterModel.addMessageItem(item);
      ChatManager.get().send(item);
    }
  }

  @Override
  public void removeFailedItem(SendingMessageItem item) {
    if (item != null) {
      adapterModel.removeMessageItem(item);
      ChatManager.get().removeFailedItem(item.getCreatedAt());
    }
  }

  @Override
  public void sendSuccessed(SendingMessageItem item, Message message) {
    adapterModel.removeMessageItem(item);
    addMessage(message);
  }

  @Override
  public void sendFailed(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public UserInfoItem getUserInfoItem() {
    return userInfoItem;
  }

  @Override
  public ProfileEntity getProfile(String personType, String personId) {
    ProfileEntity profile = dictionary.getProfile(personType, personId);
    if (profile != null) {
      return profile;
    }
    return ChannelStore.getChannel();
  }

  @Override
  public void refresh() {
    switch (state) {
      case USER_CHAT_NOT_LOADED:
        if (chatId != null) {
          fetchUserChat();
        } else {
          createUserChat();
        }
        break;

      case WAITING_SOCKET:
        SocketManager.connect();
        break;

      case NOT_JOINED:
        setState(ChatState.CHAT_JOINING, true);
        SocketManager.joinChat(chatId);
        break;

      case MESSAGES_NOT_LOADED:
        fetchMessages();
        break;
    }
  }

  private void setState(ChatState state, boolean showRefresh) {
    this.state = state;
    view.setRefreshVisibility(showRefresh);
  }

  private boolean isStateEquals(ChatState targetState) {
    return targetState.equals(state);
  }

  private void addMessageDelay(final ActionMessageItem item) {
    final Handler handler = new Handler();
    handler.postDelayed(new Runnable() {
      @Override
      public void run() {
        item.setTimestamp(forwardTimestamp);

        adapterModel.addMessageItem(item);

        view.scrollToBottom(true);
      }
    }, 750);
  }

  @Override
  public void userInfoChanged(Object object) {
    if (userInfoItem != null) {
      switch (userInfoItem.getUserInfoType()) {
        case NAME:
          if (object instanceof String) {
            userInfoItem.setObject(object);
          }
          break;

        case MOBILE_NUMBER:
          if (object instanceof MobileNumberItem) {
            userInfoItem.setObject(object);
          }
          break;
      }
    }
  }

  @Override
  public void sendUserInfo() {
    if (userInfoItem != null && userInfoItem.getObject() != null) {
      RequestUtils requestUtils = null;
      switch (userInfoItem.getUserInfoType()) {
        case NAME:
          if (userInfoItem.getObject() instanceof String) {
            requestUtils = RequestUtils.form().set("name", (String) userInfoItem.getObject());
          }
          break;

        case MOBILE_NUMBER:
          if (userInfoItem.getObject() instanceof MobileNumberItem) {
            MobileNumberItem item = (MobileNumberItem) userInfoItem.getObject();
            requestUtils = RequestUtils.form().set(
                "mobileNumber",
                String.format("+%s%s", item.getCountry(), item.getMobileNumber()));
          }
          break;

        default:
          return;
      }

      if (requestUtils != null) {
        ChannelPlugin.getApi().updateGuest(requestUtils.create())
            .subscribeOn(Schedulers.newThread())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(new RestSubscriber<UserVeilWrapper>() {
              @Override
              public void onError(Throwable e) {
                L.e(e.getMessage());

                userInfoItem.setError(true);
                adapterModel.addMessageItem(userInfoItem);
              }

              @Override
              public void onNext(UserVeilWrapper wrapper) {
                ChannelStore.setUserVeil(wrapper.getUser(), wrapper.getVeil());

                adapterModel.removeMessageItem(userInfoItem);
                checkRequireUserInfo(userInfoItem.getTimestamp());
              }
            });
      }
    }
  }

  private void checkRequireUserInfo(long timestamp) {
    boolean flag = false;

    if (userInfoItem != null) {
      try {
        adapterModel.removeMessageItem(userInfoItem);
        userInfoItem = null;
        flag = true;
      } catch (Exception ex) {
      }
    }

    if (!ChannelStore.hasName()) {
      userInfoItem = new UserInfoItem(UserInfoType.NAME, timestamp, false, null);
    } else if (ChannelStore.getMobileNumber() == null) {
      userInfoItem = new UserInfoItem(
          UserInfoType.MOBILE_NUMBER,
          timestamp,
          false,
          new MobileNumberItem(countryDictionary.getDefaultCountryCode(), ""));
    }

    if (userInfoItem != null) {
      Observable.timer(512, TimeUnit.MILLISECONDS)
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new Action1<Long>() {
            @Override
            public void call(Long aLong) {
              adapterModel.addMessageItem(userInfoItem);

              if (isUserInfoFirst) {
                isUserInfoFirst = false;
                view.scrollToBottom(true);
              }
            }
          });
    } else {
      if (flag) {
        userInfoItem = new UserInfoItem(UserInfoType.COMPLETE, timestamp, false, null);
        adapterModel.addMessageItem(userInfoItem);
      }
    }
  }

  @Override
  public void leaveChat() {
    SocketManager.leaveChat(chatId);
  }

  @Override
  public void release() {
    ChatManager.get().setOnMessageSendListener(null);
  }
}
