package com.zoyi.channel.plugin.android.activity.chat;

import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.ViewGroup;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnActionMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnSendingMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnUserInfoListener;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.presenter.chat.ChatAdapterContract;
import com.zoyi.channel.plugin.android.presenter.chat.ChatContract;
import com.zoyi.channel.plugin.android.util.CompareUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2016. 12. 8..
 */
public class ChatAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder>
    implements ChatAdapterContract.Model, ChatAdapterContract.View {

  private ChatContract.Presenter presenter;

  private SortedList<MessageItem> items;

  private OnMessageClickListener onMessageClickListener;

  private OnActionMessageClickListener onActionMessageClickListener;

  private OnSendingMessageClickListener onSendingMessageClickListener;

  private OnUserInfoListener onOnUserInfoListener;

  public ChatAdapter() {
    items = new SortedList<>(MessageItem.class, new SortedListCallback(this, false));
  }

  public void setPresenter(ChatContract.Presenter presenter) {
    this.presenter = presenter;
  }

  @Override
  public void setOnMessageClickListener(OnMessageClickListener onMessageClickListener) {
    this.onMessageClickListener = onMessageClickListener;
  }

  @Override
  public void setOnActionMessageClickListener(OnActionMessageClickListener onActionMessageClickListener) {
    this.onActionMessageClickListener = onActionMessageClickListener;
  }

  @Override
  public void setOnSendingMessageClickListener(OnSendingMessageClickListener onSendingMessageClickListener) {
    this.onSendingMessageClickListener = onSendingMessageClickListener;
  }

  public void setOnUserInfoListener(OnUserInfoListener onOnUserInfoListener) {
    this.onOnUserInfoListener = onOnUserInfoListener;
  }

  @Override
  public void addMessage(Message message, String chatId) {
    addMessages(Collections.singletonList(message), chatId);
  }

  @Override
  public void addMessages(List<Message> messages, String chatId) {
    if (messages == null) {
      return;
    }

    List<MessageItem> list = new ArrayList<>();

    for (Message message : messages) {
      if (message == null
          || message.getLog() != null
          || !CompareUtils.isSame(chatId, message.getChatId())) {
        continue;
      }

      list.add(new ChatMessageItem(message));
      list.add(new DateItem(message.getCreatedAt()));

      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        SendingMessageItem item = SendingMessageItem.create(
            message.getRequestId(),
            message.getChannelId(),
            message.getChatId());
        if (item != null) {
          items.remove(item);
        }
      }
    }

    items.addAll(list);
  }

  @Override
  public void addMessageItem(MessageItem item) {
    items.add(item);
  }

  @Override
  public void addMessageItems(Collection<MessageItem> items) {
    this.items.addAll(items);
  }

  @Override
  public void removeMessageItem(MessageItem item) {
    items.remove(item);
  }

  @Override
  public MessageItem getItem(int index) {
    if (index < 0 || index >= items.size()) {
      return null;
    }
    return items.get(index);
  }

  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    LayoutInflater i = LayoutInflater.from(parent.getContext());
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case ACTION_MESSAGE:
        return new ActionMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_manager, parent, false),
            onActionMessageClickListener);

      case QUESTION_MESSAGE:
        return new QuestionMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            presenter.getBackColor(),
            presenter.getTextColor());

      case MANAGER_MESSAGE:
        return new ManagerMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_manager, parent, false),
            onMessageClickListener);

      case SENDING:
        return new SendingMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            onSendingMessageClickListener,
            presenter.getBackColor(),
            presenter.getTextColor());

      case USER_MESSAGE:
        return new UserMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            onMessageClickListener,
            presenter.getBackColor(),
            presenter.getTextColor());

      case DATE:
        return new DateDividerHolder(
            i.inflate(R.layout.ch_plugin_item_message_date, parent, false));

      case NEW_MESSAGE_DIVIDER:
        return new NewMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_unread_divider, parent, false));

      case USER_INFO:
        return new UserInfoMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user_info, parent, false),
            onOnUserInfoListener);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case ACTION_MESSAGE:
        ActionMessageItem actionMessageItem = (ActionMessageItem) item;

        ActionMessageHolder actionMessageHolder = (ActionMessageHolder) holder;
        actionMessageHolder.set(actionMessageItem);
        break;

      case QUESTION_MESSAGE:
        QuestionMessageItem questionMessageItem = (QuestionMessageItem) item;

        QuestionMessageHolder questionMessageHolder = (QuestionMessageHolder) holder;
        questionMessageHolder.set(questionMessageItem);
        break;

      case MANAGER_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;

        ManagerMessageHolder managerMessageHolder = (ManagerMessageHolder) holder;
        managerMessageHolder.set(
            chatMessageItem,
            presenter.getProfile(
                chatMessageItem.getPersonType(),
                chatMessageItem.getPersonId()),
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case USER_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;

        UserMessageHolder userMessageHolder = (UserMessageHolder) holder;
        userMessageHolder.set(
            chatMessageItem.getMessage(),
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case SENDING:
        SendingMessageItem sendingMessageItem = (SendingMessageItem) item;
        SendingMessageHolder sendingHolder = (SendingMessageHolder) holder;
        sendingHolder.set(sendingMessageItem);
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.set((DateItem) item);
        break;

      case USER_INFO:
        UserInfoMessageHolder userInfoMessageHolder = (UserInfoMessageHolder) holder;
        userInfoMessageHolder.set(presenter.getUserInfoItem());
    }
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public int getIndex(MessageItem item) {
    return items.indexOf(item);
  }

  @Override
  public void setImageFilesToStorage() {
    PhotoAlbumStorage instance = PhotoAlbumStorage.getInstance();

    for (int i=0; i<items.size(); i++) {
      MessageItem messageItem = items.get(i);

      if (messageItem.getType() == MessageType.MANAGER_MESSAGE
          || messageItem.getType() == MessageType.USER_MESSAGE) {
        ChatMessageItem chatMessageItem = (ChatMessageItem) messageItem;
        Message message = chatMessageItem.getMessage();
        File file = message.getFile();

        if (file != null && file.isImage()) {
          if (!TextUtils.isEmpty(file.getUrl())
              && !TextUtils.isEmpty(file.getFilename())
              && file.getSize() > 0) {
            instance.addImageFile(file);
          }
        }
      }
    }
  }
}
