/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.server.frontend;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.List;

/**
 * A base class for commands providing helpers for common file generation
 * actions.
 * <p>
 * By extending this class, commands can track generated files for subsequent
 * processes. In addition, it allows to avoid writes on disk of the file already
 * exists and has exactly the same generated content, preventing file system
 * watchers to trigger unnecessary events.
 */
public abstract class AbstractFileGeneratorFallibleCommand
        implements FallibleCommand {

    private GeneratedFilesSupport generatedFilesSupport = new GeneratedFilesSupport();

    @Override
    public void setGeneratedFileSupport(GeneratedFilesSupport support) {
        this.generatedFilesSupport = support;
    }

    /**
     * Writes the given content into the given file unless the file already
     * contains that content.
     *
     * @param file
     *            the file to write to
     * @param content
     *            the lines to write
     * @return true if the content was written to the file, false otherwise
     * @throws IOException
     *             if something went wrong
     */
    protected boolean writeIfChanged(File file, List<String> content)
            throws IOException {
        return generatedFilesSupport.writeIfChanged(file, content);
    }

    /**
     * Writes the given content into the given file unless the file already
     * contains that content.
     *
     * @param file
     *            the file to write to
     * @param content
     *            the content to write
     * @return true if the content was written to the file, false otherwise
     * @throws IOException
     *             if something went wrong
     */
    protected boolean writeIfChanged(File file, String content)
            throws IOException {
        return generatedFilesSupport.writeIfChanged(file, content);
    }

    /**
     * Marks the give file as generated by the task.
     *
     * @param file
     *            the file to be marked as generated.
     */
    protected void track(File file) {
        generatedFilesSupport.track(file);
    }

    /**
     * Marks the give files as generated by the task.
     *
     * @param files
     *            the collection of files to be marked as generated.
     */
    protected void track(Collection<File> files) {
        files.forEach(generatedFilesSupport::track);
    }

}
