package com.unity3d.ads.core.domain

import android.content.Context
import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_MARKUP_URL
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.RequestType
import com.unity3d.services.core.network.model.toHttpResponse
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import gateway.v1.UniversalResponseOuterClass.UniversalResponse
import gateway.v1.adResponse

/**
 * For test purposes only. Requires VPN.
 * Converts a [UniversalResponse] with [adResponse] to a valid ad markup.
 *
 * AdMarkup is Base64 String -> convert to a ByteArray and then decode
 * example: "CgZhZERhdGEQARoSY29uZmlndXJhdGlvblRva2Vu"
 */
internal class LoadAdMarkup(
    private val defaultDispatcher: CoroutineDispatcher,
    private val getAdRequest: GetAdRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val handleGatewayAdResponse: HandleGatewayAdResponse,
    private val sessionRepository: SessionRepository,
    private val gatewayClient: GatewayClient,
    private val adRepository: AdRepository,
    private val httpClient: HttpClient,
) {
    suspend operator fun invoke(
        context: Context,
        placement: String,
        opportunityId: ByteString,
    ): String? = withContext(defaultDispatcher) {

        if (!sessionRepository.isSdkInitialized) return@withContext null

        val loadAdRequest = getAdRequest(placement, opportunityId)
        val requestPolicy = getRequestPolicy()
        val response = gatewayClient.request(
            request = loadAdRequest,
            requestPolicy = requestPolicy,
            operationType = OperationType.LOAD
        )

        val adLoadResponse = handleGatewayAdResponse(opportunityId, response.payload.adResponse, context, placement)
        if (adLoadResponse is LoadResult.Failure) return@withContext null

        adRepository.getAd(opportunityId) ?: return@withContext null

        val headers = buildMap {
            put("Content-Type", listOf("application/x-protobuf"))
        }

        val httpRequest = HttpRequest(
            baseURL = AD_MARKUP_URL,
            method = RequestType.POST,
            body = response.toByteArray(),
            headers = headers,
            isProtobuf = true
        )

        val httpResponse: HttpResponse = try {
            httpClient.execute(httpRequest)
        } catch (e: UnityAdsNetworkException) {
            e.toHttpResponse()
        }

        if (httpResponse.statusCode != 200) return@withContext null

        val httpResponseBody = httpResponse.body
        val markup = if (httpResponseBody is ByteArray) {
            httpResponseBody.toString(Charsets.ISO_8859_1)
        } else {
            return@withContext null
        }

        return@withContext markup
    }
}