package com.unity3d.ads.core.domain

import android.content.Context
import com.unity3d.ads.IUnityAdsLoadListener
import com.unity3d.ads.UnityAds.UnityAdsLoadError
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_AD_MARKUP_PARSING
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_OPPORTUNITY_ID
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_OPPORTUNITY_ID_USED
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_TIMEOUT
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.HB
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_MARKUP_PARSE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_OPPORTUNITY_ID
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_OPPORTUNITY_USED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_PLACEMENT_NULL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNCAUGHT_EXCEPTION
import com.unity3d.ads.core.extensions.*
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.UnityAdsConstants.Messages.MSG_INTERNAL_ERROR
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.request.metrics.AdOperationMetric

import headerbidding.v1.HeaderBiddingAdMarkupOuterClass.HeaderBiddingAdMarkup
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull
import java.util.UUID
import kotlin.time.ExperimentalTime
import kotlin.time.TimeMark
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
internal class LegacyLoadUseCase(
    private val dispatcher: CoroutineDispatcher,
    private val load: Load,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getInitializationState: GetInitializationState,
    private val sessionRepository: SessionRepository,
    private val adRepository: AdRepository,
) {
    private var isHeaderBidding: Boolean = false

    suspend operator fun invoke(
        context: Context,
        placement: String?,
        loadOptions: UnityAdsLoadOptions,
        unityLoadListener: IUnityAdsLoadListener?
    ) {
        val adMarkup = getAdMarkup(loadOptions)
        isHeaderBidding = !adMarkup.isNullOrBlank()

        val startTime = loadStart()
        DeviceLog.debug("Unity Ads Load Start for placement $placement")

        if (placement == null) {
            loadFailure(
                startTime = startTime,
                reason = UnityAdsLoadError.INVALID_ARGUMENT,
                diagnosticReason = REASON_PLACEMENT_NULL,
                message = LoadResult.MSG_PLACEMENT_NULL,
                placement= "",
                unityLoadListener = unityLoadListener
            )
            return
        }

        if (getInitializationState() != InitializationState.INITIALIZED) {
            loadFailure(
                startTime = startTime,
                reason = UnityAdsLoadError.INITIALIZE_FAILED,
                diagnosticReason = REASON_NOT_INITIALIZED,
                message = LoadResult.MSG_NOT_INITIALIZED,
                placement = placement,
                unityLoadListener = unityLoadListener,
            )
            return
        }

        val opportunityId = getOpportunityId(loadOptions)
        if (opportunityId == null) {
            loadFailure(
                startTime = startTime,
                reason = UnityAdsLoadError.INVALID_ARGUMENT,
                diagnosticReason = REASON_OPPORTUNITY_ID,
                message = MSG_OPPORTUNITY_ID,
                placement = placement,
                unityLoadListener = unityLoadListener
            )
            return
        }

        val opportunityIdByteString = UUID.fromString(opportunityId).toByteString()
        if (adRepository.hasOpportunityId(opportunityIdByteString)) {
            loadFailure(
                startTime = startTime,
                reason = UnityAdsLoadError.INVALID_ARGUMENT,
                diagnosticReason = REASON_OPPORTUNITY_USED,
                message = MSG_OPPORTUNITY_ID_USED,
                placement = placement,
                unityLoadListener = unityLoadListener
            )
            return
        }

        // For header bidding, the ad markup is passed in the load options and contains the adData
        val headerBiddingAdMarkup = if (!adMarkup.isNullOrBlank()) {
            try {
                HeaderBiddingAdMarkup.parseFrom(adMarkup.fromBase64().toByteArray())
            } catch (e: Exception) {
                loadFailure(
                    startTime = startTime,
                    reason = UnityAdsLoadError.INTERNAL_ERROR,
                    diagnosticReason = REASON_AD_MARKUP_PARSE,
                    message = MSG_AD_MARKUP_PARSING,
                    placement = placement,
                    unityLoadListener = unityLoadListener
                )
                return
            }

        } else {
            HeaderBiddingAdMarkup.getDefaultInstance()
        }

        val useTimeout = true // sessionRepository.nativeConfiguration.featureFlags.loadTimeoutEnabled
        val timeoutMillis = sessionRepository.nativeConfiguration.adOperations.loadTimeoutMs.toLong()

        try {
            val loadResult = if (useTimeout) {
                withTimeoutOrNull(timeoutMillis) {
                    load(context, placement, opportunityIdByteString, headerBiddingAdMarkup)
                } ?: LoadResult.Failure(
                    error = UnityAdsLoadError.TIMEOUT,
                    message = MSG_TIMEOUT + placement,
                    reason = REASON_TIMEOUT
                )
            } else {
                load(context, placement, opportunityIdByteString, headerBiddingAdMarkup)
            }

            when(loadResult) {
                is LoadResult.Success -> loadSuccess(startTime, placement, unityLoadListener)
                is LoadResult.Failure -> loadFailure(startTime, loadResult.error, loadResult.reason, loadResult.message ?: "", placement, unityLoadListener, loadResult.reasonDebug)
            }
        } catch (e: Exception) {
            loadFailure(startTime, UnityAdsLoadError.INTERNAL_ERROR, REASON_UNCAUGHT_EXCEPTION, MSG_INTERNAL_ERROR, placement, unityLoadListener, e.message)
        }

    }

    private fun getOpportunityId(unityAdsLoadOptions: UnityAdsLoadOptions): String? {
        return unityAdsLoadOptions.data?.opt(KEY_OBJECT_ID)?.toString()
    }

    private fun getAdMarkup(unityAdsLoadOptions: UnityAdsLoadOptions): String? {
        return unityAdsLoadOptions.data?.opt(KEY_AD_MARKUP)?.toString()
    }

    private fun loadStart(): TimeMark {
        val startTime = TimeSource.Monotonic.markNow()
        sendDiagnosticEvent(event = SendDiagnosticEvent.LOAD_STARTED, tags = getTags())
        return startTime
    }

    private suspend fun loadSuccess(
        startTime: TimeMark,
        placement: String,
        unityLoadListener: IUnityAdsLoadListener?
    ) {
        DeviceLog.debug("Unity Ads Load Success for placement: $placement")
        sendDiagnosticEvent(event = SendDiagnosticEvent.LOAD_SUCCESS, value = startTime.elapsedMillis(), tags = getTags())
        withContext(dispatcher) {
            unityLoadListener?.onUnityAdsAdLoaded(placement)
        }
    }

    private suspend fun loadFailure(
        startTime: TimeMark,
        reason: UnityAdsLoadError,
        diagnosticReason: String,
        message: String = "",
        placement: String,
        unityLoadListener: IUnityAdsLoadListener?,
        diagnosticMessage: String? = null,
    ) {
        DeviceLog.debug("Unity Ads Load Failure for placement: $placement reason: $reason :: $message")
        val event = if (reason == UnityAdsLoadError.TIMEOUT) SendDiagnosticEvent.LOAD_TIMEOUT else SendDiagnosticEvent.LOAD_FAILURE
        sendDiagnosticEvent(event = event, value = startTime.elapsedMillis(), tags = getTags(diagnosticReason, diagnosticMessage))
        withContext(dispatcher) {
            unityLoadListener?.onUnityAdsFailedToLoad(placement, reason, message)
        }
    }

    private fun getTags(diagnosticReason: String? = null, diagnosticMessage: String? = null): Map<String, String> {
        val tags = mutableMapOf(
            AdOperationMetric.INIT_STATE to getInitializationState().toString(),
            OPERATION to OperationType.LOAD.toString(),
            HB to isHeaderBidding.toString()
        )
        if (!diagnosticReason.isNullOrEmpty()) tags[REASON] = diagnosticReason
        if (!diagnosticMessage.isNullOrEmpty()) tags[REASON_DEBUG] = diagnosticMessage
        return tags
    }

    companion object {
        const val KEY_OBJECT_ID = "objectId"
        const val KEY_AD_MARKUP = "adMarkup"
    }
}