package com.unity3d.ads.injection

import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update

/**
 * Contains a collection of all instances registered.
 */
class Registry {
    private val _services = MutableStateFlow<Map<EntryKey, Lazy<*>>>(emptyMap())
    val services: Map<EntryKey, Lazy<*>>
        get() = _services.value

    /**
     * Creates a singleton instance of a class and adds id to the registry
     *
     * @param named is a unique id for the class if multiple implementations exist
     */
    inline fun <reified T> single(named: String = "", noinline instance: () -> T): EntryKey {
        val key = EntryKey(named, T::class)
        val lazyInstance = lazy(instance)
        add(key, lazyInstance)
        return key
    }

    /**
     * Creates a new instance of a class each time it is requested
     *
     * @param named is a unique key for the class if multiple implementations exist
     */
    inline fun <reified T> factory(named: String = "", noinline instance: () -> T): EntryKey {
        val key = EntryKey(named, T::class)
        val lazyInstance = Factory(instance)
        add(key, lazyInstance)
        return key
    }

    /**
     * Adds instance with [EntryKey] to the registry
     *
     * @param key the [EntryKey] to be added to the registry
     * @param instance the [Lazy] instance of the class
     */
    fun <T> add(key: EntryKey, instance: Lazy<T>) {
        check(!services.containsKey(key)) { "Cannot have identical entries." }
        _services.update {
            it + mapOf(key to instance as Lazy<*>)
        }
    }

    /**
     * Retrieves an instance from the registry.
     * Throws [IllegalStateException] if it does not exist
     *
     * @param named is a unique key for the instance if multiple implementations exist
     */
    inline fun <reified T> get(named: String = ""): T {
        val key = EntryKey(named, T::class)
        val lazy = services[key] ?: throw IllegalStateException("No entry found for $key")
        return lazy.value as T
    }

    /**
     * Retrieves an instance from the registry or return null if not found
     *
     * @param named is a unique key for the instance if multiple implementations exist
     */
    inline fun <reified T> getOrNull(named: String = ""): T? {
        val key = EntryKey(named, T::class)
        val lazy = services[key] ?: return null
        return lazy.value as T
    }
}
