package com.unity3d.ads.core.domain

import android.content.Context
import com.google.protobuf.ByteString
import com.unity3d.ads.IUnityAdsLoadListener
import com.unity3d.ads.UnityAds.UnityAdsLoadError
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.AdObjectState
import com.unity3d.ads.core.data.model.InitializationState.FAILED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZING
import com.unity3d.ads.core.data.model.InitializationState.NOT_INITIALIZED
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_AD_MARKUP_PARSING
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_NOT_INITIALIZED
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_OPPORTUNITY_ID_USED
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_PLACEMENT_NULL
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_TIMEOUT
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_MARKUP_PARSE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_OPPORTUNITY_USED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_PLACEMENT_NULL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT_INITIALIZATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNCAUGHT_EXCEPTION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.STATE
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.getShortenedStackTrace
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.ads.core.extensions.toUUID
import com.unity3d.ads.core.log.Logger
import com.unity3d.services.UnityAdsConstants.Messages.MSG_INTERNAL_ERROR
import com.unity3d.services.banners.UnityBannerSize
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType.DIAGNOSTIC_AD_TYPE_BANNER
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType.DIAGNOSTIC_AD_TYPE_FULLSCREEN
import gatewayprotocol.v1.HeaderBiddingAdMarkupOuterClass.HeaderBiddingAdMarkup
import kotlinx.coroutines.withTimeoutOrNull
import java.util.UUID
import kotlin.time.ExperimentalTime
import kotlin.time.TimeMark
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
internal class LegacyLoadUseCase(
    private val load: Load,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getInitializationState: GetInitializationState,
    private val awaitInitialization: AwaitInitialization,
    private val sessionRepository: SessionRepository,
    private val adRepository: AdRepository,
    private val safeCallbackInvoke: SafeCallbackInvoke,
    private val cleanUpWhenOpportunityExpires: CleanUpWhenOpportunityExpires,
    private val logger: Logger,
) {
    private var isHeaderBidding: Boolean = false
    private var isBanner: Boolean = false
    private var listener: IUnityAdsLoadListener? = null
    private var startTime: TimeMark? = null
    private var placement: String? = null
    private var opportunity: ByteString? = null
    private var adMarkup: String? = null
    private lateinit var loadOptions: UnityAdsLoadOptions

    suspend operator fun invoke(
        context: Context,
        placement: String?,
        loadOptions: UnityAdsLoadOptions,
        unityLoadListener: IUnityAdsLoadListener?,
        bannerSize: UnityBannerSize? = null,
    ) {

        // Use opportunityId from load options if available and if not, generate one
        val opportunityId = getOpportunityId(loadOptions)
            ?: UUID.randomUUID().toString()

        logger.info("Load invoked for placement: $placement with instance id: $opportunityId")

        val timeoutMillis = sessionRepository.nativeConfiguration.adOperations.loadTimeoutMs.toLong()
        val gatewayBannerSize = getBannerSize(bannerSize)

        this.loadOptions = loadOptions
        this.adMarkup = getAdMarkup(loadOptions)

        this.isHeaderBidding = !adMarkup.isNullOrBlank()
        this.isBanner = bannerSize != null
        this.listener = unityLoadListener
        this.placement = placement

        this.startTime = loadStart(opportunityId)

        try {
            val loadResult = withTimeoutOrNull(timeoutMillis) {

                if (placement == null) {
                    return@withTimeoutOrNull LoadResult.Failure(
                        error = UnityAdsLoadError.INVALID_ARGUMENT,
                        message = MSG_PLACEMENT_NULL,
                        reason = REASON_PLACEMENT_NULL,
                    )
                }

                val opportunityIdByteString = UUID.fromString(opportunityId).toByteString()
                opportunity = opportunityIdByteString
                if (adRepository.hasOpportunityId(opportunityIdByteString)) {
                    LoadResult.Failure(
                        error = UnityAdsLoadError.INVALID_ARGUMENT,
                        message = MSG_OPPORTUNITY_ID_USED,
                        reason = REASON_OPPORTUNITY_USED,
                    )
                }

                // For header bidding, the ad markup is passed in the load options and contains the adData
                val headerBiddingAdMarkup = getHeaderBiddingAdMarkup(adMarkup)
                    ?: return@withTimeoutOrNull LoadResult.Failure(
                        error = UnityAdsLoadError.INTERNAL_ERROR,
                        message = MSG_AD_MARKUP_PARSING,
                        reason = REASON_AD_MARKUP_PARSE
                    )

                when(getInitializationState()) {
                    INITIALIZED -> load(context, placement, opportunityIdByteString, headerBiddingAdMarkup, gatewayBannerSize, loadOptions)
                    NOT_INITIALIZED, FAILED -> return@withTimeoutOrNull LoadResult.Failure(
                        error = UnityAdsLoadError.INITIALIZE_FAILED,
                        message = MSG_NOT_INITIALIZED,
                        reason = REASON_NOT_INITIALIZED,
                    )
                    INITIALIZING -> {
                        when (awaitInitialization()) {
                            INITIALIZED -> load(context, placement, opportunityIdByteString, headerBiddingAdMarkup, gatewayBannerSize, loadOptions)
                            FAILED -> LoadResult.Failure(
                                error = UnityAdsLoadError.INITIALIZE_FAILED,
                                message = MSG_NOT_INITIALIZED,
                                reason = REASON_NOT_INITIALIZED,
                            )
                            else -> LoadResult.Failure(
                                error = UnityAdsLoadError.TIMEOUT,
                                message = MSG_TIMEOUT + placement,
                                reason = REASON_TIMEOUT_INITIALIZATION
                            )
                        }
                    }
                }
            } ?: LoadResult.Failure(
                error = UnityAdsLoadError.TIMEOUT,
                message = MSG_TIMEOUT + placement,
                reason = REASON_TIMEOUT
            )
            when (loadResult) {
                is LoadResult.Success -> loadSuccess(loadResult.adObject)
                is LoadResult.Failure -> loadFailure(opportunityId = opportunityId, loadResult = loadResult)
            }
        } catch (e: Throwable) {
            val loadResult = LoadResult.Failure(
                error = UnityAdsLoadError.INTERNAL_ERROR,
                message = MSG_INTERNAL_ERROR,
                reason = REASON_UNCAUGHT_EXCEPTION,
                reasonDebug = e.getShortenedStackTrace(),
                throwable = e
            )
            loadFailure(opportunityId = opportunityId, loadResult = loadResult)
        }

    }

    private fun getHeaderBiddingAdMarkup(adMarkup: String?): HeaderBiddingAdMarkup? {
        return if (!adMarkup.isNullOrBlank()) {
            try {
                HeaderBiddingAdMarkup.parseFrom(adMarkup.fromBase64().toByteArray())
            } catch (e: Exception) {
                return null
            }
        } else {
            HeaderBiddingAdMarkup.getDefaultInstance()
        }
    }

    private fun getBannerSize(bannerSize: UnityBannerSize?) = bannerSize?.let {
        gatewayprotocol.v1.bannerSize {
            width = it.width
            height = it.height
        }
    }

    private fun getOpportunityId(unityAdsLoadOptions: UnityAdsLoadOptions): String? {
        return unityAdsLoadOptions.data?.opt(KEY_OBJECT_ID)?.toString()
    }

    private fun getAdMarkup(unityAdsLoadOptions: UnityAdsLoadOptions): String? {
        return unityAdsLoadOptions.data?.opt(KEY_AD_MARKUP)?.toString()
    }

    private fun loadStart(opportunityId: String): TimeMark {
        val startTime = TimeSource.Monotonic.markNow()
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.LOAD_STARTED,
            tags = getTags(),
            adObject = getTmpAdObject(opportunityId = opportunityId),
        )
        return startTime
    }

    private fun loadSuccess(adObject: AdObject) {
        logger.info("Successfully loaded ad ${getAdInfoString(adObject.opportunityId.toUUID().toString())}")
        adObject.state.value = AdObjectState.LOADED
        cleanUpWhenOpportunityExpires(adObject)
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.LOAD_SUCCESS,
            value = startTime?.elapsedMillis(),
            tags = getTags(),
            adObject = adObject,
        )
        safeCallbackInvoke { listener?.onUnityAdsAdLoaded(placement) }
    }

    private fun loadFailure(
        opportunityId: String,
        loadResult: LoadResult.Failure
    ) {
        logger.error("Failed to load ad ${getAdInfoString(opportunityId)}, " +
            "error: ${loadResult.error} :: ${loadResult.message}")

        sendDiagnosticEvent(
            event = SendDiagnosticEvent.LOAD_FAILURE,
            value = startTime?.elapsedMillis(),
            tags = getTags(loadResult.reason, loadResult.reasonDebug),
            adObject = getTmpAdObject(opportunityId = opportunityId, isScarAd = loadResult.isScarAd),
        )
        safeCallbackInvoke {
            listener?.onUnityAdsFailedToLoad(placement, loadResult.error, loadResult.message)
        }
    }

    private fun getTags(
        reason: String? = null,
        reasonDebug: String? = null
    ): Map<String, String> {
        val tags = mutableMapOf(
            STATE to getInitializationState().toString(),
            OPERATION to OperationType.LOAD.toString(),
        )
        if (!reason.isNullOrEmpty()) tags[REASON] = reason
        if (!reasonDebug.isNullOrEmpty()) tags[REASON_DEBUG] = reasonDebug
        return tags
    }

    private fun getAdType(): DiagnosticAdType {
        return if (isBanner) DIAGNOSTIC_AD_TYPE_BANNER else DIAGNOSTIC_AD_TYPE_FULLSCREEN
    }

    private fun getTmpAdObject(opportunityId: String, isScarAd: Boolean = false): AdObject {
        val opportunityByteString = UUID.fromString(opportunityId).toByteString()

        return AdObject(
            opportunityId = opportunityByteString,
            placementId = placement ?: "",
            trackingToken = ByteString.EMPTY,
            adPlayer = null,
            loadOptions = loadOptions,
            isHeaderBidding = isHeaderBidding,
            adType = getAdType(),
            isScarAd = isScarAd,
        )
    }

    private fun getAdInfoString(opportunityId: String?): String {
        return "for placement $placement " +
            "with instance id $opportunityId"
    }

    companion object {
        const val KEY_OBJECT_ID = "objectId"
        const val KEY_AD_MARKUP = "adMarkup"
    }
}