package com.unity3d.ads.core.domain

import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.AdObjectState
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.job
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class CleanUpWhenOpportunityExpires(
    defaultDispatcher: CoroutineDispatcher,
) {
    private val coroutineExceptionHandler = CoroutineExceptionHandler { _, throwable ->
        DeviceLog.debug("CleanUpExpiredOpportunity: ${throwable.message}")
    }

    private val coroutineScope = CoroutineScope(SupervisorJob() + defaultDispatcher + coroutineExceptionHandler)

    operator fun invoke(adObject: AdObject) {
        requireNotNull(adObject.adPlayer) {
            "AdObject does not have an adPlayer."
        }

        val job = coroutineScope.launch {
            val ttl = adObject.ttl.value ?: adObject.ttl.filterNotNull().first()
            delay(ttl)

            val shouldBeDestroyed = adObject.state.value != AdObjectState.SHOWING

            if (shouldBeDestroyed) {
                withContext(NonCancellable) {
                    adObject.state.value = AdObjectState.EXPIRED
                    adObject.adPlayer.destroy()
                }
            }
        }

        adObject.adPlayer.scope.coroutineContext.job.invokeOnCompletion {
            job.cancel()
        }
    }
}