package com.unity3d.services.store.gpbl.bridges.billingclient.v8

import com.unity3d.services.store.gpbl.BillingResultResponseCode
import com.unity3d.services.store.gpbl.bridges.BillingResultBridge
import com.unity3d.services.store.gpbl.bridges.billingclient.BillingClientAdapter
import com.unity3d.services.store.gpbl.listeners.BillingClientStateListener
import com.unity3d.services.store.gpbl.listeners.BillingInitializationListener
import com.unity3d.services.store.gpbl.listeners.ProductDetailsListener
import com.unity3d.services.store.gpbl.listeners.ProductDetailsResponseListener
import com.unity3d.services.store.gpbl.listeners.PurchasesResponseListener
import com.unity3d.services.store.gpbl.proxies.BillingClientStateListenerProxy
import com.unity3d.services.store.gpbl.proxies.ProductDetailsResponseListenerProxy
import com.unity3d.services.store.gpbl.proxies.PurchaseUpdatedListenerProxy
import com.unity3d.services.store.gpbl.proxies.PurchasesResponseListenerProxy

class BillingClientAdapter(private val billingClientBuilderBridge: BillingClientBuilderBridge): BillingClientAdapter() {
    private lateinit var billingClient: BillingClientBridge
    override fun initialize(billingInitializationListener: BillingInitializationListener) {
        if (isInitialized) {
            // Already initialized (Should be revisited with UADSDK-4373) cause technically, it should be possible to re-initialize
            billingInitializationListener.onIsAlreadyInitialized()
            return
        }
        billingClientBuilderBridge.setListener(PurchaseUpdatedListenerProxy(billingInitializationListener))
        billingClientBuilderBridge.enablePendingPurchases()
        billingClient = billingClientBuilderBridge.build() as BillingClientBridge
        startConnection(billingInitializationListener)
        _isInitialized.value = true
    }

    override fun startConnection(billingClientStateListener: BillingClientStateListener?) {
        billingClient.startConnection(BillingClientStateListenerProxy(billingClientStateListener))
    }

    override fun isFeatureSupported(purchaseType: String?): BillingResultResponseCode? {
        return billingClient.isFeatureSupported(purchaseType)
    }

    override fun isReady(): Boolean {
        return billingClient.isReady
    }

    override fun queryPurchasesAsync(
        purchaseType: String?,
        purchasesResponseListener: PurchasesResponseListener?
    ) {
        billingClient.queryPurchasesAsync(purchaseType, PurchasesResponseListenerProxy(purchasesResponseListener))
    }

    override fun queryProductDetailsAsync(
        productId: String,
        productType: String,
        listener: ProductDetailsListener
    ) {

        // Build the list of products to query
        val productList = listOf(ProductBridge.newBuilder().setProductId(productId).setProductType(productType).build())
        val queryProductDetailsParamsBridge = QueryProductDetailsParamsBridge.newBuilder().setProductList(productList).build()

        // Create a proxy listener to handle the response
        val productDetailsResponseListenerProxy = ProductDetailsResponseListenerProxy(object: ProductDetailsResponseListener {
            override fun onProductDetailsResponse(
                billingResult: BillingResultBridge,
                queryProductDetailsResults: QueryProductDetailsResultsBridge
            ) {
                val productsList = queryProductDetailsResults.getProductDetailsList()
                if (productsList.isEmpty()) {
                    // No products found (wrong product type, etc.)
                    listener.onProductDetailsResponse(billingResult, null)
                    return
                }
                // There is only 1 product in the list, so we can safely access the first element
                val originalJson = productsList[0].originalJson
                listener.onProductDetailsResponse(billingResult, originalJson?.toString())
            }
        })

        billingClient.queryProductDetailsAsync(queryProductDetailsParamsBridge, productDetailsResponseListenerProxy)
    }

    override fun getAdapterVersion(): String = "8"
}