package com.unity3d.ads.core.data.manager

import android.content.Context
import com.unity3d.ads.core.data.model.exception.LoadException
import com.unity3d.ads.core.domain.scar.CommonScarEventReceiver
import com.unity3d.ads.core.domain.scar.GmaEventData
import com.unity3d.ads.core.domain.scar.ScarTimeHackFixer
import com.unity3d.ads.core.extensions.toUnityAdFormat
import com.unity3d.scar.adapter.common.GMAEvent
import com.unity3d.scar.adapter.common.scarads.ScarAdMetadata
import com.unity3d.scar.adapter.common.scarads.UnityAdFormat
import com.unity3d.services.ads.gmascar.GMAScarAdapterBridge
import com.unity3d.services.ads.gmascar.handlers.BiddingSignalsHandler
import com.unity3d.services.ads.gmascar.listeners.IBiddingSignalsListener
import com.unity3d.services.ads.gmascar.models.BiddingSignals
import com.unity3d.services.banners.BannerView
import com.unity3d.services.banners.UnityBannerSize
import com.unity3d.services.core.di.ServiceProvider.SCAR_SIGNALS_FETCH_TIMEOUT
import com.unity3d.services.core.di.ServiceProvider.SCAR_VERSION_FETCH_TIMEOUT
import gatewayprotocol.v1.AdFormatOuterClass.AdFormat
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.flow.transformWhile
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class AndroidScarManager(
    private val scarEventReceiver: CommonScarEventReceiver,
    private val gmaBridge: GMAScarAdapterBridge,
    private val scarTimeHackFixer: ScarTimeHackFixer
) : ScarManager {

    override suspend fun getVersion(): String? {
        return withTimeoutOrNull(SCAR_VERSION_FETCH_TIMEOUT) {
            scarEventReceiver.versionFlow
                .onSubscription { gmaBridge.getVersion() }
                .first()
        }
    }

    override suspend fun getSignals(adFormat: List<AdFormat>?): BiddingSignals? {
        return withTimeoutOrNull(SCAR_SIGNALS_FETCH_TIMEOUT) {
            suspendCancellableCoroutine { continuation ->
                // Convert the AdFormat to internal UnityAdFormat AdFormat
                val unityAdFormat = adFormat?.mapNotNull { it.toUnityAdFormat().takeIf { format -> format != UnityAdFormat.UNSPECIFIED } }
                if (unityAdFormat.isNullOrEmpty()) {
                    continuation.resume(null)
                    return@suspendCancellableCoroutine
                }
                gmaBridge.getSCARBiddingSignals(
                    unityAdFormat,
                    BiddingSignalsHandler(true, object: IBiddingSignalsListener {
                        override fun onSignalsReady(signals: BiddingSignals?) {
                            continuation.resume(signals)
                        }

                        override fun onSignalsFailure(msg: String?) {
                            // This is also called when SCAR is not present.
                            continuation.resumeWithException(Exception(msg))
                        }
                    })
                )
            }
        }
    }

    override suspend fun loadAd(
        adFormat: String,
        placementId: String,
        adString: String,
        adUnitId: String,
        queryId: String,
        videoLength: Int
    ) {
        // Legacy SCAR code uses canSkip flag to determine if the ad is interstitial or rewarded
        val canSkip = adFormat.equals(UnityAdFormat.INTERSTITIAL.toString(), ignoreCase = true)
        scarEventReceiver.gmaEventFlow
            .onSubscription {
                gmaBridge.load(
                    canSkip,
                    placementId,
                    queryId,
                    adString,
                    adUnitId,
                    scarTimeHackFixer(videoLength)
                )
            }
            .first {
                (
                    it.gmaEvent in listOf(GMAEvent.AD_LOADED, GMAEvent.LOAD_ERROR)
                    &&
                    it.placementId == placementId
                )
                ||  it.gmaEvent in listOf(GMAEvent.METHOD_ERROR, GMAEvent.SCAR_NOT_PRESENT, GMAEvent.INTERNAL_LOAD_ERROR)

            }
            .takeIf { it.gmaEvent != GMAEvent.AD_LOADED  }
            ?.run { throw LoadException(0, "Error loading SCAR ad: ${errorMessage ?: gmaEvent}") }
    }

    override fun loadBannerAd(
        context: Context,
        bannerView: BannerView,
        scarAdMetadata: ScarAdMetadata,
        bannerSize: UnityBannerSize,
        opportunityId: String
    ): Flow<GmaEventData>  {
        return scarEventReceiver.gmaEventFlow
            .onStart {
                gmaBridge.loadBanner(context, bannerView, opportunityId, scarAdMetadata, bannerSize)
            }
            .filter {
                it.gmaEvent == GMAEvent.BANNER && it.opportunityId == opportunityId
            }

    }

    override fun show(placementId: String, queryId: String): Flow<GmaEventData> {
        return scarEventReceiver.gmaEventFlow
            .onSubscription { gmaBridge.show(placementId, queryId) }
            .transformWhile {
                emit(it)
                // Will stop the flow when the ad is shown or an error occurs
                it.gmaEvent !in arrayOf(GMAEvent.AD_CLOSED, GMAEvent.NO_AD_ERROR, GMAEvent.INTERSTITIAL_SHOW_ERROR, GMAEvent.REWARDED_SHOW_ERROR)
            }
    }
}
