package com.unity3d.ads.core.domain

import com.unity3d.ads.IUnityAdsTokenListener
import com.unity3d.ads.TokenConfiguration
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYNC
import com.unity3d.ads.core.data.model.InitializationState.NOT_INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZING
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.AWAITED_INIT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.COMPLETE_STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_LISTENER_NULL
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.getShortenedStackTrace
import com.unity3d.services.core.misc.Utilities.wrapCustomerListener
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
class CommonInitAwaitingGetHeaderBiddingToken(
    val getHeaderBiddingToken: GetHeaderBiddingToken,
    val sendDiagnosticEvent: SendDiagnosticEvent,
    val getInitializationState: GetInitializationState,
    val awaitInitialization: AwaitInitialization,
    val sessionRepository: SessionRepository,
    val safeCallbackInvoke: SafeCallbackInvoke
) : GetAsyncHeaderBiddingToken {
    val startTime = TimeSource.Monotonic.markNow()
    var listener: IUnityAdsTokenListener? = null
    private var didAwaitInit = false
    private var startState: InitializationState? = null

    override suspend fun invoke(tokenNumber: Int, tokenConfiguration: TokenConfiguration?, listener: IUnityAdsTokenListener?) {
        this.listener = listener
        tokenStart(tokenNumber)

        if (listener == null) {
            tokenFailure(tokenNumber = tokenNumber, reason = REASON_LISTENER_NULL, reasonDebug = "IUnityAdsTokenListener is null")
            return
        }

        if (!sessionRepository.shouldInitialize) {
            tokenFailure(tokenNumber = tokenNumber, reason = REASON_GATEWAY, reasonDebug = "!sessionRepository.shouldInitialize")
            return
        }

        val tokenTimeout = sessionRepository.nativeConfiguration.adOperations.getTokenTimeoutMs.toLong()

        withTimeoutOrNull(tokenTimeout) {
            if (getInitializationState() in arrayOf(NOT_INITIALIZED, INITIALIZING)) {
                didAwaitInit = true
                awaitInitialization()
            }
        }

        if (!sessionRepository.shouldInitialize) {
            tokenFailure(tokenNumber = tokenNumber, reason = REASON_GATEWAY, reasonDebug = "!sessionRepository.shouldInitialize")
            return
        }

        fetchToken(tokenNumber, tokenConfiguration)
    }

    private suspend fun fetchToken(tokenNumber: Int, tokenConfiguration: TokenConfiguration?) {
        var reason: String? = null
        var reasonDebug: String? = null
        val token = try {
            getHeaderBiddingToken(tokenNumber, tokenConfiguration)
        } catch (e: Exception) {
            reason = SendDiagnosticEvent.REASON_UNCAUGHT_EXCEPTION
            reasonDebug = e.getShortenedStackTrace()
            null
        }
        if (token == null) {
            tokenFailure(tokenNumber, reason, reasonDebug)
        } else {
            tokenSuccess(tokenNumber, token)
        }
    }

    private fun tokenSuccess(tokenNumber: Int, token: String) {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_SUCCESS,
            value = startTime.elapsedMillis(),
            tags = mapOf(
                SYNC to false.toString(),
                STATE to startState.toString(),
                COMPLETE_STATE to getInitializationState().toString(),
                AWAITED_INIT to didAwaitInit.toString(),
            ),
            tokenNumber = tokenNumber
        )
        safeCallbackInvoke { listener?.onUnityAdsTokenReady(token) }
    }

    private fun tokenFailure(tokenNumber: Int, reason: String?, reasonDebug: String? = null) {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_FAILURE,
            value = startTime.elapsedMillis(),
            tags = buildMap {
                put(SYNC, false.toString())
                put(STATE, startState.toString())
                put(COMPLETE_STATE, getInitializationState().toString())
                put(AWAITED_INIT, didAwaitInit.toString())
                reason?.let { put(REASON, reason) }
                reasonDebug?.let { put(REASON_DEBUG, reasonDebug) }
            },
            tokenNumber = tokenNumber
        )
        safeCallbackInvoke { listener?.onUnityAdsTokenReady(null) }
    }

    private fun tokenStart(tokenNumber: Int) {
        startState = getInitializationState()
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_STARTED,
            tags = mapOf(
                SYNC to false.toString(),
                STATE to startState.toString()
            ),
            tokenNumber = tokenNumber
        )
    }
}