package com.unity3d.ads.core.domain

import android.content.Context
import com.google.android.gms.net.CronetProviderInstaller
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.configuration.MediationTraitsMetadataReader
import com.unity3d.ads.core.configuration.MediationTraitsMetadataReader.Companion.USE_REFACTORED_HTTP_CLIENT
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_SUCCESS
import com.unity3d.services.UnityAdsConstants.DefaultUrls.HTTP_CACHE_DIR_NAME
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_HOST
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_PORT
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_HOST
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_PORT
import com.unity3d.services.core.di.ServiceProvider.HTTP_CACHE_DISK_SIZE
import com.unity3d.services.core.di.ServiceProvider.HTTP_CLIENT_FETCH_TIMEOUT
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.domain.task.ConfigFileFromLocalStorage
import com.unity3d.services.core.network.core.CronetClient
import com.unity3d.services.core.network.core.CronetEngineBuilderFactory
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.core.LegacyHttpClient
import com.unity3d.services.core.network.core.OkHttp3Client
import com.unity3d.services.core.network.core.RefactoredOkHttp3Client
import com.unity3d.services.core.network.domain.CleanupDirectory
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import okhttp3.OkHttpClient
import org.chromium.net.CronetEngine
import kotlin.coroutines.resume
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

class AndroidHttpClientProvider(
    private val configFileFromLocalStorage: ConfigFileFromLocalStorage,
    private val alternativeFlowReader: AlternativeFlowReader,
    private val dispatchers: ISDKDispatchers,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val context: Context,
    private val cronetEngineBuilderFactory: CronetEngineBuilderFactory,
    private val sessionRepository: SessionRepository,
    private val cleanupDirectory: CleanupDirectory,
    private val mediationTraitsMetadataReader: MediationTraitsMetadataReader
) : HttpClientProvider {
    @OptIn(ExperimentalTime::class)
    override suspend fun invoke(gatewaySpecific: Boolean?): HttpClient {
        val isAlternativeFlowEnabled = alternativeFlowReader()
        return if (isAlternativeFlowEnabled) {
            val startTime = TimeSource.Monotonic.markNow()
            val usingRefactoredGatewayClient = (gatewaySpecific == true && mediationTraitsMetadataReader.getBooleanTrait(USE_REFACTORED_HTTP_CLIENT) == true)
            val client = withTimeoutOrNull(HTTP_CLIENT_FETCH_TIMEOUT) {
                // Check if we should use the refactored client just for gateway
                if (usingRefactoredGatewayClient) {
                    RefactoredOkHttp3Client(dispatchers, OkHttpClient())
                } else {
                    buildNetworkClient(context, dispatchers)
                }
            }
            if (!usingRefactoredGatewayClient) {
                val diagnosticResult = if (client !is CronetClient) SYSTEM_CRONET_FAILURE else SYSTEM_CRONET_SUCCESS
                sendDiagnosticEvent(diagnosticResult, startTime.elapsedNow().toDouble(DurationUnit.MILLISECONDS))
            }
            client ?: OkHttp3Client(dispatchers, OkHttpClient(), context, sessionRepository, cleanupDirectory, alternativeFlowReader)
        } else {
            val config = runBlocking {
                runCatching { configFileFromLocalStorage(ConfigFileFromLocalStorage.Params()) }.getOrNull()?.getOrNull()
            }
            if (config?.experiments?.isOkHttpEnabled == true) {
                OkHttp3Client(dispatchers, OkHttpClient(), context, sessionRepository, cleanupDirectory, alternativeFlowReader)
            } else {
                LegacyHttpClient(dispatchers)
            }
        }
    }

    private suspend fun buildNetworkClient(
        context: Context, dispatchers: ISDKDispatchers
    ): HttpClient = suspendCancellableCoroutine { continuation ->
        CronetProviderInstaller.installProvider(context).addOnCompleteListener {
            if (it.isSuccessful) {
                val httpCacheSize = if (sessionRepository.nativeConfiguration.hasCachedAssetsConfiguration()) {
                    sessionRepository.nativeConfiguration.cachedAssetsConfiguration.maxCachedAssetSizeMb.toLong() * 1024 * 1024 // to Bytes
                } else {
                    HTTP_CACHE_DISK_SIZE
                }

                try {
                    val cronetEngine =
                        cronetEngineBuilderFactory.createCronetEngineBuilder(context)
                            .setStoragePath(buildCronetCachePath(context))
                            .enableHttpCache(CronetEngine.Builder.HTTP_CACHE_DISK, httpCacheSize) // HTTP_CACHE_DISK provides 0-RTT support
                            .enableQuic(true)
                            .addQuicHint(GATEWAY_HOST, GATEWAY_PORT, GATEWAY_PORT)
                            .addQuicHint(CDN_CREATIVES_HOST, CDN_CREATIVES_PORT, CDN_CREATIVES_PORT)
                            .build()
                    continuation.resume(CronetClient(cronetEngine, dispatchers))
                } catch (e: Throwable) {
                    continuation.resume(OkHttp3Client(dispatchers, OkHttpClient(), context, sessionRepository, cleanupDirectory, alternativeFlowReader))
                }
            } else {
                continuation.resume(OkHttp3Client(dispatchers, OkHttpClient(), context, sessionRepository, cleanupDirectory, alternativeFlowReader))
            }
        }
    }

    private fun buildCronetCachePath(context: Context): String {
        // https://chromium.googlesource.com/chromium/src/+/lkgr/components/cronet/android/java/src/org/chromium/net/impl/CronetEngineBuilderImpl.java#238
        // this directory will also be fully wiped by Cronet
        val cacheDir = context.filesDir.resolve(HTTP_CACHE_DIR_NAME)
        if (!cacheDir.exists()) {
            cacheDir.mkdirs()
        }
        return cacheDir.absolutePath
    }

}