package com.unity3d.services.core.network.core

import android.content.Context
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.model.exception.NetworkTimeoutException
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.extensions.getSHA256Hash
import com.unity3d.services.UnityAdsConstants.DefaultUrls.HTTP_CACHE_DIR_NAME
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.network.domain.CleanupDirectory
import com.unity3d.services.core.network.mapper.toOkHttpProtoRequest
import com.unity3d.services.core.network.mapper.toOkHttpRequest
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import okhttp3.Call
import okhttp3.Callback
import okhttp3.OkHttpClient
import okhttp3.Response
import okio.Okio
import java.io.File
import java.io.IOException
import java.net.SocketTimeoutException
import java.util.concurrent.TimeUnit
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.math.roundToInt

/**
 * An implementation of [HttpClient] based on OkHttp
 * Supports Http2
 */
@OptIn(FlowPreview::class)
class OkHttp3Client(
    private val dispatchers: ISDKDispatchers,
    private val client: OkHttpClient,
    private val context: Context,
    private val sessionRepository: SessionRepository,
    private val cleanupDirectory: CleanupDirectory,
    private val isAlternativeFlowReader: AlternativeFlowReader,
) : HttpClient {
    private val okHttpCache = getOkHttpCache()

    /**
     * Helper method that blocks the thread to be used for Java interaction
     *
     * @param request [HttpRequest] to be executes on the network
     * @return [HttpResponse] of the passed in [HttpRequest]
     */
    override fun executeBlocking(request: HttpRequest): HttpResponse = runBlocking(dispatchers.io) {
        execute(request)
    }

    /**
     * Executes an http network request
     *
     * @param request [HttpRequest] to be executes on the network
     * @return [HttpResponse] of the passed in [HttpRequest]
     */
    override suspend fun execute(request: HttpRequest): HttpResponse = withContext(dispatchers.io) {
        try {
            val (response, body) = makeRequest(
                request,
                request.connectTimeout.toLong(),
                request.readTimeout.toLong(),
                request.writeTimeout.toLong()
            )

            val returnedBody = if (!isAlternativeFlowReader()) {
                when (body) {
                    is File -> body.readText()
                    is ByteArray -> body.toString(Charsets.UTF_8)
                    else -> ""
                }
            } else {
                body
            }

            HttpResponse(
                statusCode = response.code(),
                headers = response.headers().toMultimap(),
                urlString = response.request().url().toString(),
                body = returnedBody ?: "",
                protocol = response.protocol().toString(),
                client = NETWORK_CLIENT_OKHTTP
            )
        } catch (e: SocketTimeoutException) {
            throw NetworkTimeoutException(
                message = MSG_CONNECTION_TIMEOUT,
                url = request.baseURL,
                client = NETWORK_CLIENT_OKHTTP
            )
        } catch (e: IOException) {
            throw UnityAdsNetworkException(
                message = MSG_CONNECTION_FAILED,
                url = request.baseURL,
                client = NETWORK_CLIENT_OKHTTP
            )
        }
    }

    /**
     * Wraps the OkHttp call callback in a coroutine with structured concurrency
     */
    private suspend fun makeRequest(
        request: HttpRequest,
        connectTimeout: Long,
        readTimeout: Long,
        writeTimeout: Long,
    ): RequestComplete {
        val okHttpRequest = if (request.isProtobuf) request.toOkHttpProtoRequest() else request.toOkHttpRequest()

        val configuredClient = client.newBuilder()
            .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
            .readTimeout(readTimeout, TimeUnit.MILLISECONDS)
            .writeTimeout(writeTimeout, TimeUnit.MILLISECONDS)
            .build()

        val filename = request.baseURL.getSHA256Hash()
        val file = File(okHttpCache, filename)
        val downloadedLength = if (file.exists() && file.isFile) file.length() else 0L

        val okHttpRequestWithRange = downloadedLength.takeIf { it > 0 }?.let {
            DeviceLog.debug("Resuming download for ${request.baseURL}")
            okHttpRequest.newBuilder()
                .addHeader("Range", "bytes=$it-")
                .build()
        }

        return suspendCancellableCoroutine { continuation ->
            configuredClient.newCall(okHttpRequestWithRange ?: okHttpRequest).enqueue(object : Callback {
                override fun onResponse(call: Call, response: Response) {
                    if (!response.isSuccessful) {
                        return continuation.resumeWithException(IOException("Network request failed with code ${response.code()}"))
                    }

                    try {
                        val body = response.body() ?: return continuation.resume(RequestComplete(response))

                        val contentLength = body.contentLength()
                        val memorySink = okio.Buffer()
                        val canCache = response.header("Cache-Control")?.contains("no-cache") == false
                        val fileSink = takeIf { canCache }?.let {
                            if (!file.exists()) {
                                file.createNewFile()
                            }
                            Okio.buffer(Okio.appendingSink(file))
                        }

                        val downloadProgressLogging = fileSink?.let { MutableStateFlow(0L) }
                        val downloadProgressLoggingJob = downloadProgressLogging
                            ?.debounce(1000)
                            ?.filter { it != 0L }
                            ?.map { bytesDownloaded -> (bytesDownloaded.toFloat() / contentLength * 100).roundToInt() }
                            ?.onEach { progress -> DeviceLog.debug("Downloaded $progress% of ${request.baseURL}") }
                            ?.launchIn(CoroutineScope(dispatchers.io))

                        val source = body.source()
                        val buffer = fileSink?.buffer() ?: memorySink.buffer()
                        generateSequence { source.read(buffer, 8192L) }
                            .takeWhile { it != -1L }
                            .fold(0L) { acc, bytesRead ->
                                (acc + bytesRead).also { bytesDownloaded ->
                                    fileSink?.emitCompleteSegments()
                                    downloadProgressLogging?.tryEmit(bytesDownloaded)
                                }
                            }

                        memorySink.close()
                        fileSink?.close()
                        downloadProgressLoggingJob?.cancel()
                        source.close()
                        body.close()
                        buffer.close()

                        val resultBody = if (fileSink != null) {
                            file
                        } else {
                            memorySink.readByteArray()
                        }

                        continuation.resume(RequestComplete(response, resultBody))
                    } catch (e: IOException) {
                        continuation.resumeWithException(e)
                    }
                }

                override fun onFailure(call: Call, e: IOException) {
                    continuation.resumeWithException(e)
                }
            })
        }
    }

    private fun getOkHttpCache(): File {
        val cacheDir = context.filesDir.resolve(HTTP_CACHE_DIR_NAME)
        cacheDir.mkdirs()

        if (sessionRepository.nativeConfiguration.hasCachedAssetsConfiguration()) {
            val config = sessionRepository.nativeConfiguration.cachedAssetsConfiguration

            cleanupDirectory(cacheDir, config.maxCachedAssetSizeMb, config.maxCachedAssetAgeMs)
        }

        return cacheDir
    }

    private data class RequestComplete(val response: Response, val body: Any? = null)

    companion object {
        const val MSG_CONNECTION_TIMEOUT = "Network request timeout"
        const val MSG_CONNECTION_FAILED = "Network request failed"
        const val NETWORK_CLIENT_OKHTTP = "okhttp"
    }
}
