package com.unity3d.ads.core.domain

import android.content.Context
import android.net.Uri
import android.webkit.WebResourceResponse
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.repository.CacheRepository
import com.unity3d.ads.core.extensions.guessMimeType
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_ASSET_DOMAIN
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_CACHE_DOMAIN
import kotlinx.coroutines.runBlocking

class GetCachedAsset(
    private val cacheRepository: CacheRepository,
    private val context: Context,
    private val cacheWebViewAssets: CacheWebViewAssets,
) {
    operator fun invoke(uri: Uri, webviewType: String = ""): WebResourceResponse? {
        return when (uri.host) {
            AD_CACHE_DOMAIN -> getCachedAsset(uri)
            AD_ASSET_DOMAIN -> getBundledAsset(uri)
            else -> tryGetWebViewAsset(uri, webviewType)
        }
    }

    private fun getBundledAsset(uri: Uri): WebResourceResponse? {
        val fileName = uri.path.toString().substringAfter("/")
        return try {
            val inputStream = context.assets.open(fileName)
            WebResourceResponse(fileName.guessMimeType(), null, inputStream)
        } catch (e: Exception) {
            null
        }
    }

    private fun getCachedAsset(uri: Uri): WebResourceResponse? {
        val urlString = uri.toString()
        val fileName = urlString.substringAfterLast("/")
        val result = runBlocking { cacheRepository.retrieveFile(fileName) }
        return when (result) {
            is CacheResult.Success -> {
                val rawFile = result.cachedFile.file ?: return null
                val fileInputStream = runCatching { rawFile.inputStream() }.getOrNull() ?: return null
                val filePath = rawFile.absolutePath
                val mimeType = filePath.guessMimeType()
                if (mimeType.isNullOrBlank()) return null
                WebResourceResponse(mimeType, null, fileInputStream)
            }
            else -> null
        }
    }

    private fun tryGetWebViewAsset(uri: Uri, webviewType: String): WebResourceResponse? {
        val filename = "$webviewType/${uri.toString().substringBefore("?").substringAfterLast("/")}"
        val file = cacheWebViewAssets.cached[filename]
        return if (file != null) {
            val fileInputStream = runCatching { file.inputStream() }.getOrNull() ?: return null
            val mimeType = filename.guessMimeType()
            if (mimeType.isNullOrBlank()) return null
            WebResourceResponse(mimeType, null, fileInputStream)
        } else {
            null
        }
    }
}