package com.unity3d.ads.core.domain

import android.content.Context
import android.net.Uri
import android.webkit.WebResourceResponse
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.repository.CacheRepository
import com.unity3d.ads.core.extensions.guessMimeType
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_ASSET_DOMAIN
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_CACHE_DOMAIN



class GetCachedAsset(
    private val cacheRepository: CacheRepository,
    private val context: Context,
) {
    operator fun invoke(uri: Uri): WebResourceResponse? {
        return when (uri.host) {
            AD_CACHE_DOMAIN -> getCachedAsset(uri)
            AD_ASSET_DOMAIN -> getBundledAsset(uri)
            else -> null
        }
    }

    private fun getBundledAsset(uri: Uri): WebResourceResponse? {
        val fileName = uri.path.toString().substringAfter("/")
        return try {
            val inputStream = context.assets.open(fileName)
            WebResourceResponse(fileName.guessMimeType(), null, inputStream)
        } catch (e: Exception) {
            null
        }
    }

    private fun getCachedAsset(uri: Uri): WebResourceResponse? {
        val urlString = uri.toString()
        val file = urlString.substringAfterLast("/")
        val fileName = file.substringBeforeLast(".")
        return when (val result = cacheRepository.retrieveFile(fileName)) {
            is CacheResult.Success -> {
                val rawFile = result.cachedFile.file ?: return null
                val fileInputStream = runCatching { rawFile.inputStream() }.getOrNull() ?: return null
                val filePath = "${rawFile.absolutePath}.${result.cachedFile.extension}"
                val mimeType = filePath.guessMimeType()
                if (mimeType.isNullOrBlank()) return null
                WebResourceResponse(filePath.guessMimeType(), null, fileInputStream)
            }
            else -> null
        }
    }
}