package com.unity3d.services.store

import com.unity3d.ads.core.extensions.toJsonArray
import com.unity3d.services.store.gpbl.BillingResultResponseCode
import com.unity3d.services.store.gpbl.bridges.BillingResultBridge
import com.unity3d.services.store.gpbl.bridges.PurchaseBridge
import com.unity3d.services.store.gpbl.bridges.PurchaseHistoryRecordBridge
import com.unity3d.services.store.gpbl.bridges.SkuDetailsBridge
import com.unity3d.services.store.gpbl.listeners.StoreEventListener
import org.json.JSONArray

class WebViewStoreEventListener(
    override val operationId: Int,
    private val storeWebViewEventSender: StoreWebViewEventSender,
    private val isLifecycleListener: Boolean = false
): StoreEventListener {
    override fun onIsAlreadyInitialized() {
        storeWebViewEventSender.send(
            StoreEvent.INITIALIZATION_REQUEST_RESULT,
            BillingResultResponseCode.OK.responseCode
        )
    }

    override fun onFeatureSupported(featureSupported: Int) {
        storeWebViewEventSender.send(
            StoreEvent.IS_FEATURE_SUPPORTED_REQUEST_RESULT,
            operationId,
            featureSupported
        )
    }

    override fun onBillingSetupFinished(billingResult: BillingResultBridge) {
        if (billingResult.responseCode == BillingResultResponseCode.OK) {
            storeWebViewEventSender.send(
                StoreEvent.INITIALIZATION_REQUEST_RESULT,
                billingResult.responseCode
            )
        } else {
            storeWebViewEventSender.send(
                StoreEvent.INITIALIZATION_REQUEST_FAILED,
                billingResult.responseCode
            )
        }
    }

    override fun onBillingServiceDisconnected() {
        storeWebViewEventSender.send(StoreEvent.DISCONNECTED_RESULT)
    }

    override fun onPurchaseHistoryUpdated(
        billingResult: BillingResultBridge,
        purchaseHistoryRecordList: List<PurchaseHistoryRecordBridge>?
    ) {
        storeWebViewEventSender.send(
            StoreEvent.PURCHASE_HISTORY_LIST_REQUEST_RESULT,
            operationId,
            billingResult.responseCode,
            purchaseHistoryRecordList?.toJsonArray() ?: JSONArray()
        )
    }

    override fun onSkuDetailsUpdated(
        billingResult: BillingResultBridge,
        skuDetailsList: List<SkuDetailsBridge>?
    ) {
        storeWebViewEventSender.send(
            StoreEvent.SKU_DETAILS_LIST_REQUEST_RESULT,
            operationId,
            skuDetailsList?.toJsonArray() ?: JSONArray()
        )
    }

    override fun onPurchaseResponse(
        billingResult: BillingResultBridge,
        purchases: List<PurchaseBridge>?
    ) {
        val params = ArrayList<Any>()
        if (operationId != -1) {
            // We provide operation ID only when the request would have come from the WebView.
            params.add(operationId)
        }
        if (billingResult.responseCode == BillingResultResponseCode.OK) {
            val purchasesJson = purchases?.toJsonArray() ?: JSONArray()
            params.add(purchasesJson)
            storeWebViewEventSender.send(if (isLifecycleListener) StoreEvent.PURCHASES_ON_RESUME_RESULT else StoreEvent.PURCHASES_REQUEST_RESULT, *params.toTypedArray())
        } else {
            params.add(billingResult.responseCode)
            storeWebViewEventSender.send(if (isLifecycleListener) StoreEvent.PURCHASES_ON_RESUME_ERROR else StoreEvent.PURCHASES_REQUEST_ERROR, *params.toTypedArray())
        }
    }

    override fun onPurchaseUpdated(
        billingResult: BillingResultBridge,
        purchases: List<PurchaseBridge>?
    ) {
        if (billingResult.responseCode == BillingResultResponseCode.OK) {
            storeWebViewEventSender.send(
                StoreEvent.PURCHASES_UPDATED_RESULT,
                purchases?.toJsonArray() ?: JSONArray()
            )
        } else {
            storeWebViewEventSender.send(
                StoreEvent.PURCHASES_UPDATED_ERROR,
                billingResult.responseCode
            )
        }
    }


}