package com.unity3d.ads.core.data.datasource

import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.domain.CreateFile
import com.unity3d.ads.core.domain.GetFileExtensionFromUrl
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.isSuccessful
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import java.io.File

class AndroidRemoteCacheDataSource(
    private val ioDispatcher: CoroutineDispatcher,
    private val createFile: CreateFile,
    private val getFileExtensionFromUrl: GetFileExtensionFromUrl,
    private val httpClient: HttpClient
) : CacheDataSource {
    override suspend fun getFile(
        cachePath: File,
        fileName: String,
        url: String?,
        priority: Int?
    ): CacheResult {
        if (url == null) {
            return CacheResult.Failure(CacheError.MALFORMED_URL, CacheSource.REMOTE)
        }

        val response = try {
            downloadFile(url, priority)
        } catch (e: Exception) {
            return CacheResult.Failure(CacheError.NETWORK_ERROR, CacheSource.REMOTE)
        }
        val extension = if (url.isEmpty()) null else getFileExtensionFromUrl(url)

        return if (response.isSuccessful()) {
            val file = createFile(cachePath, fileName)

            withContext(ioDispatcher) {
                file.createNewFile()
                file.writeBytes(response.body as ByteArray)
            }

            val cachedFile = CachedFile(
                objectId = "",
                url = url,
                name = fileName,
                file = file,
                extension = extension ?: "",
                contentLength = response.contentSize,
                protocol = response.protocol,
                priority = priority ?: Int.MAX_VALUE
            )

            CacheResult.Success(cachedFile, CacheSource.REMOTE)
        } else {
            CacheResult.Failure(CacheError.NETWORK_ERROR, CacheSource.REMOTE)
        }
    }

    private suspend fun downloadFile(url: String, priority: Int?): HttpResponse {
        val request = HttpRequest(
            baseURL = url,
            priority = priority ?: Int.MAX_VALUE
        )
        return httpClient.execute(request)
    }
}