package com.unity3d.services.core.network.core

import com.unity3d.services.core.domain.ISDKDispatchers
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import okio.BufferedSink
import okio.Okio
import org.chromium.net.CronetException
import org.chromium.net.UrlRequest
import org.chromium.net.UrlResponseInfo
import java.io.ByteArrayOutputStream
import java.io.File
import java.nio.ByteBuffer
import java.nio.channels.Channels

internal abstract class UnityAdsUrlRequestCallback(private val dispatchers: ISDKDispatchers, val readTimeout: Long, val file: File?) : UrlRequest.Callback() {
    private val bytesReceived = ByteArrayOutputStream()
    private val receiveChannel = Channels.newChannel(bytesReceived)
    private lateinit var sink: BufferedSink
    private var task: Job? = null

    override fun onRedirectReceived(request: UrlRequest, info: UrlResponseInfo?, newLocationUrl: String?) {
        request.followRedirect()
    }

    final override fun onResponseStarted(request: UrlRequest, info: UrlResponseInfo) {
        if (file?.exists() == true) {
            sink = Okio.buffer(Okio.sink(file))
        }

        startTimer(request)
        request.read(ByteBuffer.allocateDirect(BYTE_BUFFER_CAPACITY_BYTES))
    }

    final override fun onReadCompleted(
        request: UrlRequest, info: UrlResponseInfo, byteBuffer: ByteBuffer
    ) {
        cancelTimer()

        byteBuffer.flip()
        if (file?.exists() == true) {
            sink.write(byteBuffer)
        } else {
            receiveChannel.write(byteBuffer)
        }
        byteBuffer.clear()
        startTimer(request)
        request.read(byteBuffer)
    }

    final override fun onSucceeded(request: UrlRequest, info: UrlResponseInfo) {
        cancelTimer()

        val bodyBytes = bytesReceived.toByteArray()
        if (file != null && file.exists()) {
            sink.close()
        }
        onSucceeded(request, info, bodyBytes)
    }

    override fun onCanceled(request: UrlRequest?, info: UrlResponseInfo?) {
        super.onCanceled(request, info)

        cancelTimer()
    }

    override fun onFailed(request: UrlRequest?, info: UrlResponseInfo?, error: CronetException?) {
        cancelTimer()
    }

    abstract fun onSucceeded(request: UrlRequest, info: UrlResponseInfo, bodyBytes: ByteArray)

    fun startTimer(request: UrlRequest) {
        cancelTimer()

        task = CoroutineScope(dispatchers.io).launch {
            delay(readTimeout)
            request.cancel()
        }
    }

    private fun cancelTimer() {
        task?.cancel()
        task = null
    }

    companion object {
        private const val BYTE_BUFFER_CAPACITY_BYTES = 16 * 1024
    }
}