package com.unity3d.ads.core.data.datasource

import com.unity3d.ads.core.domain.privacy.FlattenerRulesUseCase
import com.unity3d.services.core.misc.JsonFlattener
import com.unity3d.services.core.misc.JsonStorage
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsent
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentChoice
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentChoice.DEVELOPER_CONSENT_CHOICE_FALSE
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentChoice.DEVELOPER_CONSENT_CHOICE_TRUE
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentChoice.DEVELOPER_CONSENT_CHOICE_UNSPECIFIED
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentOption
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_CUSTOM
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_GDPR_CONSENT
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_NON_BEHAVIORAL
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_PIPL_CONSENT
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_PRIVACY_CONSENT
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_UNSPECIFIED
import gatewayprotocol.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_USER_OVER_AGE_LIMIT
import gatewayprotocol.v1.developerConsent
import gatewayprotocol.v1.developerConsentOption
import org.json.JSONObject

class AndroidDeveloperConsentDataSource(
    private val flattenerRulesUseCase: FlattenerRulesUseCase,
    private val publicStorage: JsonStorage,
) : DeveloperConsentDataSource {
    override val developerConsent: DeveloperConsent
        get() = developerConsent {
            options.addAll(developerConsentList())
        }

    private fun developerConsentList(): List<DeveloperConsentOption> {
        val developerConsentList = mutableListOf<DeveloperConsentOption>()
        val data = fetchData()

        data.keys().forEach { key ->
            val storedValue = data.get(key)
            val consentOption = when (storedValue) {
                is Boolean -> createDeveloperConsentOption(key, storedValue)
                is String -> if (storedValue.equals("true", ignoreCase = true) || storedValue.equals("false", ignoreCase = true)) {
                    createDeveloperConsentOption(key, storedValue.toBoolean())
                } else null
                else -> null
            }
            consentOption?.let { developerConsentList.add(it) }
        }

        return developerConsentList
    }

    private fun createDeveloperConsentOption(key: String, value: Boolean): DeveloperConsentOption {
        return developerConsentOption {
            type = getDeveloperConsentType(key)
            if (type == DEVELOPER_CONSENT_TYPE_CUSTOM) {
                customType = key
            }
            this.value = getDeveloperConsentChoice(value)
        }
    }

    private fun fetchData(): JSONObject {
        publicStorage.data?.let {
            val flattener = JsonFlattener(publicStorage.data)
            return flattener.flattenJson(".", flattenerRulesUseCase())
        }?: return JSONObject()
    }

    private fun getDeveloperConsentType(type: String?): DeveloperConsentType {
        return when (type) {
            "user.nonBehavioral" -> DEVELOPER_CONSENT_TYPE_NON_BEHAVIORAL
            "user.nonbehavioral" -> DEVELOPER_CONSENT_TYPE_NON_BEHAVIORAL
            "pipl.consent" -> DEVELOPER_CONSENT_TYPE_PIPL_CONSENT
            "privacy.consent" -> DEVELOPER_CONSENT_TYPE_PRIVACY_CONSENT
            "gdpr.consent" -> DEVELOPER_CONSENT_TYPE_GDPR_CONSENT
            "privacy.useroveragelimit" -> DEVELOPER_CONSENT_TYPE_USER_OVER_AGE_LIMIT
            null -> DEVELOPER_CONSENT_TYPE_UNSPECIFIED
            else -> DEVELOPER_CONSENT_TYPE_CUSTOM
        }
    }

    private fun getDeveloperConsentChoice(choice: Boolean?): DeveloperConsentChoice {
        return when (choice) {
            true -> DEVELOPER_CONSENT_CHOICE_TRUE
            false -> DEVELOPER_CONSENT_CHOICE_FALSE
            else -> DEVELOPER_CONSENT_CHOICE_UNSPECIFIED
        }
    }
}