package com.unity3d.ads.core.domain.work

import android.content.Context
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import com.unity3d.ads.core.data.datasource.UniversalRequestDataSource
import com.unity3d.ads.core.domain.events.UniversalRequestEventSender
import com.unity3d.ads.gatewayclient.RequestPolicy
import com.unity3d.services.core.di.IServiceComponent
import com.unity3d.services.core.di.inject
import gatewayprotocol.v1.UniversalRequestOuterClass.UniversalRequest

abstract class UniversalRequestJob(
    context: Context,
    protected val workerParams: WorkerParameters
) : CoroutineWorker(context, workerParams), IServiceComponent {
    lateinit var requestPolicy: RequestPolicy

    val universalRequestEventSender: UniversalRequestEventSender by inject()
    val universalRequestDataSource: UniversalRequestDataSource by inject()

    override suspend fun doWork(): Result {
        val universalRequestId =
            workerParams.inputData.getString(UniversalRequestWorkerData.KEY_UNIVERSAL_REQUEST_ID)?: return Result.success()
        val rawUniversalRequest = universalRequestDataSource.get().getUniversalRequestMapOrThrow(universalRequestId)
        val universalRequest = UniversalRequest.parseFrom(rawUniversalRequest)
        val universalRequestEventSender = runCatching {
            universalRequestEventSender(universalRequest, requestPolicy)
        }
        return if (universalRequestEventSender.isSuccess) {
            universalRequestDataSource.remove(universalRequestId)
            Result.success()
        } else {
            Result.retry()
        }
    }
}