package com.unity3d.ads.core.domain

import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_COMPLETED_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_COMPLETED_STARTED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_COMPLETED_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNCAUGHT_EXCEPTION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNKNOWN
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_REQ


import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)

class TriggerAndroidInitializationCompletedRequest(
    private val getInitializationCompletedRequest: GetInitializationCompletedRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val gatewayClient: GatewayClient,
    private val sendDiagnosticEvent: SendDiagnosticEvent
) : TriggerInitializationCompletedRequest {
    override suspend fun invoke() {
        val startTime = TimeSource.Monotonic.markNow()
        sendDiagnosticEvent(event = INIT_COMPLETED_STARTED)
        try {
            val initializationCompletedRequest = getInitializationCompletedRequest()
            val requestPolicy = getRequestPolicy()
            val response = gatewayClient.request(
                request = initializationCompletedRequest,
                requestPolicy = requestPolicy,
                operationType = OperationType.INITIALIZATION_COMPLETED
            )
            if (response.hasError()) {
                sendDiagnosticEvent(
                    event = INIT_COMPLETED_FAILURE,
                    value = startTime.elapsedMillis(),
                    tags = mapOf(
                        OPERATION to OperationType.INITIALIZATION_COMPLETED.toString(),
                        REASON to REASON_GATEWAY,
                        REASON_DEBUG to response.error.errorText
                    )
                )
            }
        } catch (e: Exception) {
            sendDiagnosticEvent(
                event = INIT_COMPLETED_FAILURE,
                value = startTime.elapsedMillis(),
                tags = mapOf(
                    OPERATION to OperationType.INITIALIZATION_COMPLETED.toString(),
                    REASON to REASON_UNCAUGHT_EXCEPTION,
                    REASON_DEBUG to (e.message ?: REASON_UNKNOWN)
                )
            )
        }
        sendDiagnosticEvent(event = INIT_COMPLETED_SUCCESS, value = startTime.elapsedMillis())
    }
}