package com.unity3d.services

import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.COROUTINE_NAME
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.extensions.retrieveUnityCrashValue
import com.unity3d.services.core.di.ServiceProvider.IO_DISPATCHER
import com.unity3d.services.core.di.ServiceProvider.NAMED_SDK
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.request.metrics.Metric
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import org.koin.core.annotation.Named
import org.koin.core.annotation.Single
import kotlin.coroutines.CoroutineContext

@Single
@Named(NAMED_SDK)
class SDKErrorHandler(
    @Named(IO_DISPATCHER) private val ioDispatcher: CoroutineDispatcher,
    private val alternativeFlowReader: AlternativeFlowReader,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val sdkMetricsSender: SDKMetricsSender
) : CoroutineExceptionHandler {
    private val scope = CoroutineScope(ioDispatcher) + CoroutineName("SDKErrorHandler")

    override val key = CoroutineExceptionHandler.Key

    override fun handleException(context: CoroutineContext, exception: Throwable) {
        val coroutineName = retrieveCoroutineName(context)
        val name: String = when (exception) {
            is NullPointerException -> "native_exception_npe"
            is OutOfMemoryError -> "native_exception_oom"
            is IllegalStateException -> "native_exception_ise"
            is SecurityException -> "native_exception_se"
            is RuntimeException -> "native_exception_re"
            else -> "native_exception"
        }

        val isAlternativeFlowEnabled = alternativeFlowReader()
        val crashValue = exception.retrieveUnityCrashValue()
        DeviceLog.error("Unity Ads SDK encountered an exception: $crashValue")
        if (isAlternativeFlowEnabled) {
            sendDiagnostic(name, crashValue, coroutineName)
        } else {
            sendMetric(Metric(name, crashValue))
        }

    }

    private fun sendDiagnostic(name: String, reason: String, scopeName: String) {
        scope.launch {
            sendDiagnosticEvent(
                event = name,
                tags = mapOf(
                    REASON to reason,
                    COROUTINE_NAME to scopeName
                )
            )
        }
    }

    private fun sendMetric(metric: Metric) = sdkMetricsSender.sendMetric(metric)

    private fun retrieveCoroutineName(context: CoroutineContext) = context[CoroutineName]?.name ?: "unknown"

    companion object {
        const val UNKNOWN_FILE = "unknown"
        const val UNITY_PACKAGE = "com.unity3d"
    }
}