package com.unity3d.ads.core.extensions

import com.google.protobuf.ByteString
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.util.UUID

fun UUID.toByteString(): ByteString {
    val uuidBytes = ByteArray(16)
    val bytes =  ByteBuffer.wrap(uuidBytes)
        .order(ByteOrder.BIG_ENDIAN)
        .putLong(this.mostSignificantBits)
        .putLong(this.leastSignificantBits)
    return ByteString.copyFrom(bytes.array())
}

fun ByteString.toUUID(): UUID {
    val bytebuffer: ByteBuffer = this.asReadOnlyByteBuffer()
    // In case of a 36 byte ByteString, we assume it's a UUID string and try to parse it
    if (bytebuffer.remaining() == 36) {
        val uuidString = this.toStringUtf8()
        return UUID.fromString(uuidString)
    } else if (bytebuffer.remaining() != 16) {
        throw IllegalArgumentException("Expected 16 byte ByteString or UUID string")
    }
    val high: Long = bytebuffer.long
    val low: Long = bytebuffer.long
    return UUID(high, low)
}

fun ByteString.toBase64(): String {
    return android.util.Base64.encodeToString(this.toByteArray(), android.util.Base64.NO_WRAP)
}

fun String.fromBase64(): ByteString {
    return ByteString.copyFrom(android.util.Base64.decode(this, android.util.Base64.NO_WRAP))
}

/**
 * Converts a [String] to a [ByteString] using [Charsets.ISO_8859_1] encoding.
 * This encoding is a 1-to-1 mapping between bytes and characters and avoids the errors created
 * when converting back and forth using UTF-8 via {@link ByteStrings#toByteStringUtf8()}.
 */
fun String.toISO8859ByteString(): ByteString {
    return ByteString.copyFrom(this.toByteArray(Charsets.ISO_8859_1))
}

/**
 * Converts a [ByteString] to a [String] using [Charsets.ISO_8859_1] encoding.
 * This encoding is a 1-to-1 mapping between bytes and characters and avoids the errors created
 * when converting back and forth using UTF-8 via {@link ByteStrings#toStringUtf8()}.
 */
fun ByteString.toISO8859String(): String {
    return this.toString(Charsets.ISO_8859_1)
}