package com.unity3d.ads.core.data.datasource

import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.services.core.di.ServiceProvider
import com.unity3d.services.core.di.ServiceProvider.NAMED_REMOTE
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.isSuccessful
import org.koin.core.annotation.Named
import org.koin.core.annotation.Single
import java.io.File

@Single
@Named(NAMED_REMOTE)
class AndroidRemoteCacheDataSource(
    private val httpClient: HttpClient
) : CacheDataSource {

    override suspend fun getFile(
        cachePath: File,
        fileName: String,
        url: String?,
        priority: Int?
    ): CacheResult {
        if (url == null) {
            return CacheResult.Failure(CacheError.MALFORMED_URL, CacheSource.REMOTE)
        }
        val file = setupFile(cachePath, fileName)
        val response = downloadFile(url, file, priority)
        val urlWithoutQuery = url.substringBefore("?")
        val extension = urlWithoutQuery.substringAfterLast(".")
        val cachedFile = CachedFile(
            objectId = "",
            url = url,
            name = fileName,
            file = file,
            extension = extension,
            contentLength = response.contentSize,
            protocol = response.protocol,
            priority = priority ?: Int.MAX_VALUE
        )

        return if (response.isSuccessful()) {
            CacheResult.Success(cachedFile, CacheSource.REMOTE)
        } else {
            CacheResult.Failure(CacheError.NETWORK_ERROR, CacheSource.REMOTE)
        }
    }

    private suspend fun downloadFile(url: String, destination: File, priority: Int?): HttpResponse {
        val request = HttpRequest(
            baseURL = url,
            downloadDestination = destination,
            priority = priority ?: Int.MAX_VALUE
        )
        return httpClient.execute(request)
    }

    private fun setupFile(cachePath: File, filename: String): File {
        val file = File(cachePath, filename)
        if (!file.exists()) {
            file.createNewFile()
        } else {
            file.delete()
            file.createNewFile()
        }
        return file
    }
}