package com.unity3d.ads.core.domain

import android.util.Base64
import com.google.protobuf.ByteString
import com.google.protobuf.kotlin.toByteString
import com.unity3d.ads.adplayer.ExposedFunction
import com.unity3d.ads.adplayer.ExposedFunctionLocation
import com.unity3d.ads.adplayer.ExposedFunctionLocation.*
import com.unity3d.ads.adplayer.Invocation
import com.unity3d.ads.core.domain.om.GetOmData
import com.unity3d.ads.core.domain.om.OmFinishSession
import com.unity3d.ads.core.domain.om.OmImpressionOccurred
import com.unity3d.ads.core.domain.om.OmInteraction
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.repository.CampaignRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.domain.om.IsOMActivated
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.ads.core.utils.ContinuationFromCallback
import com.unity3d.services.UnityAdsConstants.Cache.CACHE_SCHEME
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_CACHE_DOMAIN
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_JS_URL
import com.unity3d.services.core.api.Storage
import gateway.v1.OperativeEventRequestOuterClass.OperativeEventType
import gateway.v1.PrivacyUpdateRequestOuterClass.PrivacyUpdateRequest
import gateway.v1.copy
import kotlinx.coroutines.flow.*
import org.json.JSONArray
import org.json.JSONObject
import kotlin.coroutines.suspendCoroutine

internal class HandleInvocationsFromAdViewer(
    private val getAndroidAdPlayerContext: GetAndroidAdPlayerContext,
    private val getOperativeEventApi: GetOperativeEventApi,
    private val refresh: Refresh,
    private val handleOpenUrl: HandleOpenUrl,
    private val sessionRepository: SessionRepository,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val campaignRepository: CampaignRepository,
    private val sendPrivacyUpdateRequest: SendPrivacyUpdateRequest,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val cacheFile: CacheFile,
    private val getIsFileCached: GetIsFileCache,
    private val omStartSession: OmInteraction,
    private val omFinishSession: OmFinishSession,
    private val omImpressionOccurred: OmImpressionOccurred,
    private val getOMData: GetOmData,
    private val isOMActivated: IsOMActivated,
) {
    suspend operator fun invoke(
        onInvocations: SharedFlow<Invocation>,
        adData: String,
        adDataRefreshToken: String,
        impressionConfig: String,
        adObject: AdObject,
        onSubscription: suspend () -> Unit,
    ): Flow<Invocation> {
        val exposedFunctions = mapOf<ExposedFunctionLocation, ExposedFunction>(
            GET_AD_CONTEXT to {
                val isOMActivated = isOMActivated()
                buildMap {
                    put(KEY_AD_DATA, adData)
                    put(KEY_IMPRESSION_CONFIG, impressionConfig)
                    put(KEY_AD_DATA_REFRESH_TOKEN, adDataRefreshToken)
                    put(KEY_NATIVE_CONTEXT, getAndroidAdPlayerContext())
                    if (isOMActivated) {
                        put(KEY_OMJS, OM_JS_URL)
                    }
                }
            },
            OM_START_SESSION to {
                val options = it[0] as JSONObject
                omStartSession(adObject, options)
            },
            OM_FINISH_SESSION to { omFinishSession(adObject) },
            OM_IMPRESSION to {
                val signalLoaded = it[0] as Boolean
                omImpressionOccurred(adObject, signalLoaded)
            },
            OM_GET_DATA to {
                val data = getOMData()
                buildMap {
                    put(KEY_OM_VERSION, data.version)
                    put(KEY_OM_PARTNER, data.partnerName)
                    put(KEY_OM_PARTNER_VERSION, data.partnerVersion)
                }
            },
            GET_SCREEN_HEIGHT to { deviceInfoRepository.staticDeviceInfo().screenHeight },
            GET_SCREEN_WIDTH to { deviceInfoRepository.staticDeviceInfo().screenWidth },
            GET_CONNECTION_TYPE to { deviceInfoRepository.connectionTypeStr },
            GET_DEVICE_VOLUME to { deviceInfoRepository.dynamicDeviceInfo.android.volume },
            GET_DEVICE_MAX_VOLUME to { deviceInfoRepository.dynamicDeviceInfo.android.maxVolume },
            SEND_OPERATIVE_EVENT to {
                getOperativeEventApi(
                    adObject = adObject,
                    operativeEventType = OperativeEventType.OPERATIVE_EVENT_TYPE_SPECIFIED_BY_AD_PLAYER,
                    additionalEventData = Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
                )
            },
            OPEN_URL to {
                val url = it[0] as String
                val params = it.getOrNull(1) as? JSONObject
                val packageName = params?.optString(KEY_PACKAGE_NAME)

                handleOpenUrl(url, packageName)
            },
            STORAGE_WRITE to {
                suspendCoroutine { continuation ->
                    Storage.write(
                        it[0] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            STORAGE_CLEAR to {
                suspendCoroutine { continuation ->
                    Storage.clear(
                        it[0] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            STORAGE_DELETE to {
                suspendCoroutine { continuation ->
                    Storage.delete(
                        it[0] as String,
                        it[1] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            STORAGE_READ to {
                suspendCoroutine { continuation ->
                    Storage.read(
                        it[0] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            STORAGE_GET_KEYS to {
                suspendCoroutine { continuation ->
                    Storage.getKeys(
                        it[0] as String,
                        it[1] as String,
                        it[2] as Boolean,
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            STORAGE_GET to {
                suspendCoroutine { continuation ->
                    Storage.get(
                        it[0] as String,
                        it[1] as String,
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            STORAGE_SET to {
                suspendCoroutine { continuation ->
                    Storage.set(
                        it[0] as String,
                        it[1] as String,
                        it[2],
                        ContinuationFromCallback(continuation)
                    )
                }
            },
            GET_PRIVACY_FSM to { sessionRepository.getPrivacyFsm().toBase64() },
            SET_PRIVACY_FSM to {
                sessionRepository.setPrivacyFsm(
                    Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
                )
            },
            SET_PRIVACY to {
                sessionRepository.setPrivacy(
                    Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
                )
            },
            GET_PRIVACY to { sessionRepository.getPrivacy().toBase64() },
            GET_ALLOWED_PII to {
                Base64.encodeToString(deviceInfoRepository.allowedPii.value.toByteArray(), Base64.NO_WRAP)
            },
            SET_ALLOWED_PII to {
                val allowedPiiUpdated = it[0] as JSONObject

                deviceInfoRepository.allowedPii.update { allowedPii ->
                    allowedPii.copy {
                        allowedPiiUpdated.optBoolean("idfa")?.let(::idfa::set)
                        allowedPiiUpdated.optBoolean("idfv")?.let(::idfv::set)
                    }
                }
            },
            GET_SESSION_TOKEN to { sessionRepository.sessionToken.toBase64() },
            MARK_CAMPAIGN_STATE_SHOWN to { campaignRepository.setShowTimestamp(adObject.opportunityId) },
            REFRESH_AD_DATA to {
                val refreshTokenByteString = if (it.isEmpty()) {
                    ByteString.EMPTY
                } else {
                    val refreshTokenJson = it[0] as JSONObject
                    val refreshToken = refreshTokenJson.optString(KEY_AD_DATA_REFRESH_TOKEN)
                    refreshToken.fromBase64()
                }

                val adRefreshResponse = refresh(refreshTokenByteString, adObject.opportunityId)

                if (adRefreshResponse.hasError()) throw IllegalArgumentException("Refresh failed")

                buildMap {
                    put(KEY_AD_DATA, adRefreshResponse.adData.toBase64())
                    put(KEY_AD_DATA_REFRESH_TOKEN, adRefreshResponse.adDataRefreshToken.toBase64())
                    put(KEY_TRACKING_TOKEN, adRefreshResponse.trackingToken.toBase64())
                }
            },
            UPDATE_TRACKING_TOKEN to {
                val updateTrackingToken = it[0] as JSONObject
                val token = updateTrackingToken.optString("trackingToken")

                if (!token.isNullOrEmpty()) {
                    adObject.trackingToken = token.fromBase64()
                }
            },
            SEND_PRIVACY_UPDATE_REQUEST to {
                val base64Proto = it[0] as String
                val privacyUpdateRequest =
                    PrivacyUpdateRequest.parseFrom(Base64.decode(base64Proto, Base64.NO_WRAP))

                val response = sendPrivacyUpdateRequest(privacyUpdateRequest)

                Base64.encodeToString(response.toByteArray(), Base64.NO_WRAP)
            },
            SEND_DIAGNOSTIC_EVENT to {
                val event = it[0] as String
                val tags = it[1] as JSONObject
                val tagsMap = buildMap {
                    tags.keys().forEach { key ->
                        put(key, tags.getString(key))
                    }
                }
                val value = it.getOrNull(2)?.toString()?.toDouble()
                sendDiagnosticEvent(event = event, value = value, tags = tagsMap)
            },
            INCREMENT_BANNER_IMPRESSION_COUNT to {
                sessionRepository.incrementBannerImpressionCount()
            },
            DOWNLOAD to {
                val json = it[0] as JSONObject
                val url = json.getString(KEY_DOWNLOAD_URL)
                val headers = it.getOrNull(2) as JSONArray?
                val priority = json.optInt(KEY_DOWNLOAD_PRIORITY, 0)

                val result = cacheFile(url, adObject.opportunityId, headers, priority)
                when (result) {
                    is CacheResult.Success -> "$CACHE_SCHEME://$AD_CACHE_DOMAIN/${result.cachedFile.name}.${result.cachedFile.extension}"
                    is CacheResult.Failure -> throw IllegalStateException("Download failed")
                }

            },
            IS_FILE_CACHED to {
                val fileUrl = it[0] as String
                getIsFileCached(fileUrl)
            },
        )

        return onInvocations
            .onSubscription { onSubscription() }
            .onEach {
                val exposedFunction = exposedFunctions[it.location] ?: return@onEach

                it.handle { exposedFunction(it.parameters) }
            }
    }

    companion object {
        const val KEY_AD_DATA = "adData"
        const val KEY_AD_DATA_REFRESH_TOKEN = "adDataRefreshToken"
        const val KEY_TRACKING_TOKEN = "trackingToken"
        const val KEY_IMPRESSION_CONFIG = "impressionConfig"
        const val KEY_NATIVE_CONTEXT = "nativeContext"
        const val KEY_PACKAGE_NAME = "packageName"
        const val KEY_DOWNLOAD_URL = "url"
        const val KEY_DOWNLOAD_PRIORITY = "priority"
        const val KEY_OMJS = "omidFileUrl"
        const val KEY_OM_VERSION = "omidVersion"
        const val KEY_OM_PARTNER = "omidPartner"
        const val KEY_OM_PARTNER_VERSION = "omidPartnerVersion"
        const val KEY_OM_ACTIVE_SESSIONS = "omidActionSessions"
    }
}