/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.util;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.log4j.Logger;
import org.elasticsearch.log4j.message.ParameterizedMessage;

public class IndexFolderUpgrader {
    private final NodeEnvironment nodeEnv;
    private final Settings settings;
    private final Logger logger = Loggers.getLogger(IndexFolderUpgrader.class);

    IndexFolderUpgrader(Settings settings, NodeEnvironment nodeEnv) {
        this.settings = settings;
        this.nodeEnv = nodeEnv;
    }

    void upgrade(Index index, Path source, Path target) throws IOException {
        boolean success = false;
        try {
            Files.move(source, target, StandardCopyOption.ATOMIC_MOVE);
            success = true;
        }
        catch (FileNotFoundException | NoSuchFileException exception) {
            this.logger.error(() -> new ParameterizedMessage("multiple nodes trying to upgrade [{}] in parallel, retry upgrading with single node", (Object)target), (Throwable)exception);
            throw exception;
        }
        finally {
            if (success) {
                this.logger.info("{} moved from [{}] to [{}]", (Object)index, (Object)source, (Object)target);
                this.logger.trace("{} syncing directory [{}]", (Object)index, (Object)target);
                IOUtils.fsync(target, true);
            }
        }
    }

    void upgrade(String indexFolderName) throws IOException {
        for (NodeEnvironment.NodePath nodePath : this.nodeEnv.nodePaths()) {
            Path indexFolderPath = nodePath.indicesPath.resolve(indexFolderName);
            IndexMetaData indexMetaData = IndexMetaData.FORMAT.loadLatestState(this.logger, NamedXContentRegistry.EMPTY, indexFolderPath);
            if (indexMetaData != null) {
                Index index = indexMetaData.getIndex();
                if (IndexFolderUpgrader.needsUpgrade(index, indexFolderName)) {
                    this.logger.info("{} upgrading [{}] to new naming convention", (Object)index, (Object)indexFolderPath);
                    IndexSettings indexSettings = new IndexSettings(indexMetaData, this.settings);
                    if (indexSettings.hasCustomDataPath()) {
                        Path customLocationSource = this.nodeEnv.resolveBaseCustomLocation(indexSettings).resolve(indexFolderName);
                        Path customLocationTarget = customLocationSource.resolveSibling(index.getUUID());
                        if (Files.exists(customLocationSource, new LinkOption[0]) && !Files.exists(customLocationTarget, new LinkOption[0])) {
                            this.upgrade(index, customLocationSource, customLocationTarget);
                        } else {
                            this.logger.info("[{}] no upgrade needed - already upgraded", (Object)customLocationTarget);
                        }
                    }
                    this.upgrade(index, indexFolderPath, indexFolderPath.resolveSibling(index.getUUID()));
                    continue;
                }
                this.logger.debug("[{}] no upgrade needed - already upgraded", (Object)indexFolderPath);
                continue;
            }
            this.logger.warn("[{}] no index state found - ignoring", (Object)indexFolderPath);
        }
    }

    public static void upgradeIndicesIfNeeded(Settings settings, NodeEnvironment nodeEnv) throws IOException {
        IndexFolderUpgrader upgrader = new IndexFolderUpgrader(settings, nodeEnv);
        for (String indexFolderName : nodeEnv.availableIndexFolders()) {
            upgrader.upgrade(indexFolderName);
        }
    }

    static boolean needsUpgrade(Index index, String indexFolderName) {
        return !indexFolderName.equals(index.getUUID());
    }
}

