package com.tts.android.aopmonitor.aspect;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.view.View;

import com.tts.android.aopmonitor.AOPMonitorUtil;
import com.tts.android.aopmonitor.R;
import com.tts.android.aopmonitor.RecordService;
import com.tts.android.aopmonitor.annotation.NoBindMonitor;
import com.tts.android.aopmonitor.annotation.PageStatistics;
import com.tts.android.aopmonitor.bean.ViewClickAction;
import com.tts.android.aopmonitor.tools.Constants;
import com.tts.android.aopmonitor.tools.LogUtils;
import com.tts.android.aopmonitor.tools.TimeUtils;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;

/**
 * view的onclick方法的切入
 * 加入点击控件信息的收集
 * 并把内容存入
 * @author shijianting
 * create 2018/8/16 下午1:26
 */
@Aspect
public class ClickRecordAspect {


    /**
     * the special aspect location
     * use for the action which can not be auto aspect
     */
    private final String POINT_CONTENT = "execution(@com.tts.android.aopmonitor.annotation.BindMonitor * *(..))";

    /**
     * aspect the click action of view
     */
    private final String POINT_ALL_ONCLICK = "execution(* android.view.View.OnClickListener.onClick(..))";

    /**
     * aspect the itemclick of collection widgets
     */
    private final String POINT_ALL_ITEMONCLICK = "execution(* android.widget.AdapterView.OnItemClickListener.onItemClick(..))";

    /**
     * aspect the check change action of the checkbox
     */
    private final String POINT_ALL_CHECK = "execution(* android.widget.CompoundButton.OnCheckedChangeListener.onCheckedChanged(..))";



    @Pointcut(POINT_CONTENT)
    public void unKnownClickMethod(){}

    @Pointcut(POINT_ALL_ONCLICK)
    public void allOnClickMethod(){}

    @Pointcut(POINT_ALL_ITEMONCLICK)
    public void allItemOnClickMethod(){}

    @Pointcut(POINT_ALL_CHECK)
    public void allCheckChangeMethod(){}


    /**
     * click action of view
     * @param point
     * @throws Throwable
     */
    @After("allOnClickMethod()")
    public void clickRecord(JoinPoint point) throws Throwable {

        if (!isEnabledMonitor(point)) {
            return;
        }

        View v = getArgView(point);

        executeRecord(v);

    }

    /**
     * itemClick action of adapter
     * @param point
     * @throws Throwable
     */
    @After("allItemOnClickMethod()")
    public void itemClickRecord(JoinPoint point) throws Throwable{

        if (!isEnabledMonitor(point)) {
            return;
        }

        View v = getArgView(point);

        executeRecord(v);

    }


    @After("allCheckChangeMethod()")
    public void checkChangeRecord(JoinPoint point) throws Throwable{

        if (!isEnabledMonitor(point)) {
            return;
        }

        View v = getArgView(point);

        Boolean checked = null;

        Object[] args = point.getArgs();
        for (Object obj:args
                ) {
            if (obj instanceof Boolean) {
                checked = (Boolean) obj;
            }
        }

        if (null != checked) {
            if (null != v.getTag(R.id.aopmonitor_extend) && !TextUtils.isEmpty(v.getTag(R.id.aopmonitor_extend).toString())) {
                String extend = v.getTag(R.id.aopmonitor_extend).toString();
                //防止extend累加
                extend = extend.replace("true", "");
                extend = extend.replace("false", "");
                extend = extend.replace("&", "");
                extend += checked.booleanValue()+"";
                AOPMonitorUtil.setExtendData(v, extend);
            }
            else {
                AOPMonitorUtil.setExtendData(v, checked.booleanValue()+"");
            }
        }

        executeRecord(v);
    }

    /**
     * unidentified view click action record
     * @param point
     */
    @After("unKnownClickMethod()")
    public void unknownViewClickRecord(JoinPoint point) {

        if (!isEnabledMonitor(point)) {
            return;
        }

        View v = getArgView(point);

        executeRecord(v);
    }

    /**
     * 是否启用监控
     * @param point 切入点
     * @return
     */
    public boolean isEnabledMonitor(JoinPoint point) {
        //这是反射，使用的地方注意不要被混淆
        MethodSignature methodSignature = (MethodSignature) point.getSignature();
        Method method = methodSignature.getMethod();
        //不执行统计流程
        if (!Constants.enabledMonitor) {
            return false;
        }
        if (null == method) {
            LogUtils.loge("can't get the method by reflect");
            return true;
        }
        //获取不注入代码注解
        //加入这个注解，不执行插入代码
        Object aspectFlag = method.getAnnotation(NoBindMonitor.class);
        if (aspectFlag != null) {
            return false;
        }
        return true;
    }

    /**
     * 执行记录操作
     * @param v 搭载数据的view
     */
    public void executeRecord(View v) {
        String extend = "";
        if (null != v) {
            try {
                //使用view里面的tag-extend属性
                if (null != v.getTag(R.id.aopmonitor_extend)) {
                    extend = v.getTag(R.id.aopmonitor_extend).toString();
                }
                Context c = v.getContext();
                //控件的内容，contentDescription属性
                String description = (String) v.getContentDescription();
                //page
                String page = getPageName(c);

                //控件的最终id
                String viewId = "";

                if(!TextUtils.isEmpty(description)) {
                    viewId = description;
                }
                //控件没有id
                else if(v.getId() != -1) {
                    viewId = c.getResources().getResourceEntryName(v.getId());
                }
                else {
                    LogUtils.loge(String.format("this %s have no id, in %s ! \n it's a necessary param",
                            v.getClass().getName().toString(), page));
                }
                ViewClickAction action = getClickAction(page, viewId, extend);
                sendAction(c, action);
            }catch (Exception e) {
                e.printStackTrace();
            }
        }
        else {
            LogUtils.loge("the method has no view param or the params are incorrect");
        }
    }

    /**
     * 根据Activity context获取相应的页面名称
     * @param context
     * @return
     */
    public static String getPageName(Context context) {
        String pageName = "";
        //先获取注解配置的名称-优先级高
        PageStatistics bindPageName = context.getClass().getAnnotation(PageStatistics.class);
        if (null != bindPageName && !TextUtils.isEmpty(bindPageName.value())) {
            pageName = bindPageName.value();
            return pageName;
        }
        else if (null != bindPageName && bindPageName.name() != -1) {
            pageName = context.getString(bindPageName.name());
            return  pageName;
        }
        //获取activity的label-优先级中
        if (null != context && context instanceof Activity) {
            if (null != ((Activity) context).getTitle()) {
                pageName = String.valueOf(((Activity) context).getTitle());
            }
            if (!TextUtils.isEmpty(pageName)) {
                return pageName;
            }
        }
        //直接使用页面完整类名-优先级低
        pageName = ((Activity)context).getComponentName().getClassName();
        return pageName;
    }

    /**
     * 发送手机的数据广播
     * @param context activity的引用
     * @param action 动作
     */
    public void sendAction(Context context, ViewClickAction action) {
        Intent intent = new Intent();
        intent.setAction(RecordService.RECORD_TAG);
        intent.putExtra(RecordService.ACTION_TYPE, RecordService.ACTION_RECORD);
        intent.putExtra(RecordService.DATA, action);
        context.sendBroadcast(intent);
    }

    /**
     * 从参数数组中获取view参数
     * @param point
     * @return
     */
    public View getArgView(JoinPoint point) {

        //方法中携带的所有参数
        Object[] args = point.getArgs();
        for (Object obj:args
                ) {
            if (obj instanceof View) {
                return (View) obj;
            }
        }

        return null;
    }

    /**
     * 生成action对象
     * @param pageId
     * @param widgetId
     * @param extend
     * @return
     */
    public ViewClickAction getClickAction(String pageId,
                                          String widgetId,
                                          String extend) {
        ViewClickAction action = new ViewClickAction();
        action.setWidgetId(TextUtils.isEmpty(widgetId) ? "" : widgetId);
        action.setPageId(TextUtils.isEmpty(pageId) ? "" : pageId);
        action.setExtend(TextUtils.isEmpty(extend) ? "" : extend);
        action.setTimeStamp(TimeUtils.getTimeStamp());
        return action;
    }

}
