package com.tts.android.aopmonitor;

import android.annotation.SuppressLint;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.ServiceConnection;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.blankj.utilcode.util.AppUtils;
import com.blankj.utilcode.util.CacheDiskUtils;
import com.blankj.utilcode.util.DeviceUtils;
import com.blankj.utilcode.util.PhoneUtils;
import com.google.gson.Gson;
import com.tts.android.aopmonitor.bean.ViewClickAction;
import com.tts.android.aopmonitor.net.NetManager;
import com.tts.android.aopmonitor.net.bean.RecordRequestBean;
import com.tts.android.aopmonitor.tools.Constants;
import com.tts.android.aopmonitor.tools.CtwUtils;
import com.tts.android.aopmonitor.tools.LogUtils;
import com.tts.android.aopmonitor.tools.TimeUtils;

import java.util.ArrayList;

public class RecordService extends Service{

    private volatile static RecordService service;

    private BroadcastReceiver receiver;

    private ArrayList<ViewClickAction> actions = new ArrayList<>();

    private RecordSendHandler handler;

    private final int HANDLER_TAG = 90001;

    /**
     * 记录广播标示
     */
    public static final String RECORD_TAG = "com.record";
    public static final String ACTION_TYPE = "type";
    public static final String DATA = "data";
    public static final String PAGE = "page";

    public static final int ACTION_RECORD = 1001;

    public static final int PAGE_RECORD = 1002;

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        //放入这里的内容要注意，可能会执行多次
        return super.onStartCommand(intent, flags, startId);
    }

    @Override
    public void onCreate() {
        super.onCreate();
        service = this;
        LogUtils.logd("RecordService启动了");
        handler = new RecordSendHandler();
        /**
         * 操作のレコードを受け取る
         */
        receiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                int type = intent.getIntExtra(ACTION_TYPE, 0);
                switch (type) {
                    case ACTION_RECORD:
                        //用户行为数据记录
                        ViewClickAction action = (ViewClickAction) intent.getSerializableExtra(DATA);
                        if (null != action) {
                            actions.add(action);
                            checkActions();
                        }
                        break;
                    case PAGE_RECORD:
                        //页面跳转记录
                        String page = intent.getStringExtra(PAGE);
                        RecordRequestBean bean = new RecordRequestBean();
                        bean = initBaseRecordBean(bean);
                        bean.setPageId(page);
                        //对象转String，加入基础数据
                        String actionStr = new Gson().toJson(CtwUtils.initCtwRecordBean(bean));
                        putOneActionInHandler(actionStr);
                        break;
                    default:

                        break;
                }

            }
        };
        IntentFilter filter = new IntentFilter();
        filter.addAction(RECORD_TAG);
        registerReceiver(receiver, filter);
        resendDiskRecord();
    }


    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public boolean bindService(Intent service, ServiceConnection conn, int flags) {
        return super.bindService(service, conn, flags);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        service = null;
        LogUtils.logd("RecordService关闭了");
        unregisterReceiver(receiver);
    }

    /**
     * 使用队列管理网络请求发送
     */
    private class RecordSendHandler extends Handler {

        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            if (msg.what == HANDLER_TAG && null != msg.obj) {
                String action = msg.obj.toString();
                NetManager.getInstance(RecordService.this).doRecord(action);
            }
        }
    }

    /**
     * 重新发送缓存在磁盘，上次发送失败的记录
     */
    private void resendDiskRecord() {
        if (CacheDiskUtils.getInstance().getCacheCount() > 0) {
            int i = 0;
            //未发送的记录
            while (true) {
                String action = CacheDiskUtils.getInstance().getString("r"+i, "");
                if (!TextUtils.isEmpty(action)) {
                    putOneActionInHandler(action);
                    CacheDiskUtils.getInstance().remove("r"+i);
                }
                else {
                    break;
                }
                i++;
            }
            //crash日志
            String crashInfo = CacheDiskUtils.getInstance().getString("crash", "");
            if (!TextUtils.isEmpty(crashInfo)) {
                putOneActionInHandler(crashInfo);
                CacheDiskUtils.getInstance().remove("crash");
            }
            LogUtils.logd("disk cache record have been send");
        }
        else {
            LogUtils.logd("have no disk cache records");
        }
    }

    /**
     * 检查动作条数，根据配置判断是否发送数据
     */
    private void checkActions() {
        if (actions.size() >= Constants.moniterOption.getActionCount()) {
            RecordRequestBean bean = new RecordRequestBean();
            bean = initBaseRecordBean(bean);
            bean.setActions(actions);
            //对象转String，加入基础数据
            String actionStr = new Gson().toJson(CtwUtils.initCtwRecordBean(bean));
            putOneActionInHandler(actionStr);
            actions.clear();
        }
    }

    /**
     * 把一条数据放入消息队列
     * @param action
     */
    private void putOneActionInHandler(String action) {
        Message msg = Message.obtain();
        msg.what = HANDLER_TAG;
        msg.obj = action;
        handler.sendMessage(msg);
    }

    /**
     * 初始化监控数据外层数据内容
     * @param bean
     * @return
     */
    @SuppressLint("MissingPermission")
    public static RecordRequestBean initBaseRecordBean(RecordRequestBean bean) {
        bean.setAppVersionCode(AppUtils.getAppVersionCode()+"");
        bean.setAppVersionName(AppUtils.getAppVersionName());
        bean.setFlowCode(Constants.random32String);
        bean.setUniqueId(PhoneUtils.getDeviceId());
        bean.setTimeStamp(TimeUtils.getTimeStamp());
        bean.setOrderSource("android");
        bean.setChannel("1");
        bean.setUserId(Constants.userId);
        bean.setLatitude(Constants.latitude);
        bean.setAltitude(Constants.altitude);
        bean.setSystemVersion(DeviceUtils.getSDKVersionName());
        bean.setManufacturer(Build.MANUFACTURER);
        bean.setModel(Build.MODEL);
        bean.setResolution(Build.DISPLAY);
        return bean;
    }

    /**
     * 服务是否存活
     *
     * @return true: 存活
     */
    public static boolean isServiceRunning() {
        return service != null;
    }

}
