package com.tts.android.aopmonitor.net;

import android.content.Context;
import android.text.TextUtils;

import com.blankj.utilcode.util.CacheDiskUtils;
import com.tts.android.aopmonitor.BuildConfig;
import com.tts.android.aopmonitor.options.MoniterOption;
import com.tts.android.aopmonitor.tools.Constants;
import com.tts.android.aopmonitor.tools.FileUtil;
import com.tts.android.aopmonitor.tools.LogUtils;
import com.tts.android.aopmonitor.tools.PropertiesUtil;
import com.tts.android.aopmonitor.tools.TimeUtils;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.ref.SoftReference;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Converter;
import retrofit2.Response;
import retrofit2.Retrofit;

/**
 * 网络请求管理器
 * @author shijianting
 * 2018/11/27 8:55 AM
 */
public class NetManager {

    Context mContext;

    private static final String CTW = "ctw";

    /**
     * 存储发送记录失败的本地缓存的id
     */
    public static int diskCacheId = 0;

    public static String baseUrl = null;

    static Retrofit retrofit;

    private static volatile NetManager instance;

    public static NetManager getInstance(Context context) {
        if (instance == null) {
            synchronized (NetManager.class) {
                if (instance == null) {
                    instance = new NetManager(context);
                }
            }
        }
        return instance;

    }

    public NetManager(Context context) {
        baseUrl = Constants.moniterOption.getUrl();
        mContext = context.getApplicationContext();
        initRetrofit();
    }

    /**
     * 初始化retrofit
     * @return
     */
    private static boolean initRetrofit() {
        if (TextUtils.isEmpty(baseUrl)) {
            LogUtils.loge("baseUrl is null, init retrofit failed");
            return false;
        }
        retrofit = new Retrofit.Builder()
                .baseUrl(baseUrl)
                //使用自定义string解析器
                .addConverterFactory(new Converter.Factory() {

                    @Override
                    public Converter<ResponseBody, ?> responseBodyConverter(Type type, Annotation[] annotations, Retrofit retrofit) {
                        return new Converter<ResponseBody, String>() {
                            @Override
                            public String convert(ResponseBody value) throws IOException {
                                return value.string();
                            }
                        };
                    }
                })
                .build();
        return true;
    }

    /**
     * send monitor request
     * @param actions
     */
    public void doRecord(final String actions) {
        //发送前检查
        if (!checkNet()) {
            saveRecordInDiskCache(actions);
            return;
        }
        RequestRecord requestRecord = retrofit.create(RequestRecord.class);
        HashMap<String, String> params = initDefaultMParams();
        params.put("data", actions);
        Call<String> call = null;
        //本地记录
        if (Constants.moniterOption.isDishCache()) {
            LogUtils.logd("write to Monitor file");
            FileUtil.writeStringInFile("Monitor", actions);
        }
        //如果传入标志为ctw，进入畅途网专用网络请求流程
        if (CTW.equals(Constants.moniterOption.getDataKey())) {
            //拼装Url
            String url = Constants.moniterOption.getUrl() + Constants.moniterOption.getCmd() + "?" + actions;
            OkHttpClient client = new OkHttpClient.Builder()
                    .retryOnConnectionFailure(true)
                    .connectTimeout(15, TimeUnit.SECONDS)
                    .build();
            Request request = new Request.Builder()
                    .url(url)
                    .get()
                    .build();
            okhttp3.Call callctw = client.newCall(request);
            //打印日志
            LogUtils.logd(url);
            LogUtils.logjson(actions);
            callctw.enqueue(new okhttp3.Callback() {
                @Override
                public void onFailure(okhttp3.Call call, IOException e) {
                    LogUtils.logd(e.getMessage());
                    saveRecordInDiskCache(actions);
                }

                @Override
                public void onResponse(okhttp3.Call call, okhttp3.Response response) throws IOException {
                    LogUtils.logd(response.message());
                }
            });
            return;
        }
        LogUtils.logjson(actions);
        //request type
        if (Constants.moniterOption.getRequestType() == MoniterOption.REQUESTTYPE_POST) {
            call = requestRecord.postActionSend(Constants.moniterOption.getCmd(), params);
        }
        else if (Constants.moniterOption.getRequestType() == MoniterOption.REQUESTTYPE_GET) {
            call = requestRecord.getActionSend(Constants.moniterOption.getCmd(), params);
        }
        else {
            return;
        }
        call.enqueue(new Callback<String>() {

            @Override
            public void onResponse(Call<String> call, Response<String> response) {
                LogUtils.logd(response.body());
            }

            @Override
            public void onFailure(Call<String> call, Throwable t) {
                LogUtils.logd(t.getMessage());
                saveRecordInDiskCache(actions);
            }
        });

    }

    /**
     * 发送记录前的检查
     * @return 是否通过检查
     */
    private boolean checkNet() {
        //检查Url是否存在
        if (TextUtils.isEmpty(baseUrl)) {
            LogUtils.loge("baseUrl is null");
            return false;
        }
        //检查retrofit是否初始化
        if (null == retrofit) {
            boolean result = initRetrofit();
            if (!result) {
                LogUtils.loge("reInit retrofit failed");
                return false;
            }
        }
        //检查是否配置了dataKey
        if (TextUtils.isEmpty(Constants.moniterOption.getDataKey())) {
            LogUtils.loge("dataKey is null");
            return false;
        }
        return true;
    }

    /**
     * 发送失败后，保存数据到本地缓存
     * 在服务下次启动的时候发送出去
     * 重启后的再次磁盘缓存会覆盖掉原来未提交的缓存记录
     * @param actions
     */
    private void saveRecordInDiskCache(String actions) {
        LogUtils.logd("save action in disk cache r"+diskCacheId);
        CacheDiskUtils.getInstance().put("r"+diskCacheId, actions);
        synchronized (this) {
            diskCacheId++;
        }
    }

    /**
     * 伪装参数名
     * @param paramJson
     * @return
     */
    @Deprecated
    private String disguiseParamNames(String paramJson) {
        String newParamJson = paramJson;
        //获取外层参数的伪装map
        HashMap<String,String> recordKeyMap = PropertiesUtil.getAllProperties(mContext, "recordKeyMap");
        if (recordKeyMap.size() > 0) {
            for (Map.Entry<String, String> entry : recordKeyMap.entrySet()) {
                newParamJson = newParamJson.replace(entry.getKey(), entry.getValue());
            }
        }
        return  newParamJson;
    }

    /**
     * 初始化请求参数
     * @return
     */
    private HashMap<String, String> initDefaultMParams() {
        HashMap<String, String> params = new HashMap<>(3);
        params.put("flowCode", Constants.random32String);
        params.put("timeStamp", TimeUtils.getTimeStamp());
        params.put("appVersionCode", BuildConfig.VERSION_CODE + "");
        return params;
    }
}
