/******************************************************************************
 * 		
 * 	 Copyright 2015, TRX Systems, Inc.  All Rights Reserved.
 * 
 *   TRX Systems, Inc.
 *   7500 Greenway Center Drive, Suite 420
 *   Greenbelt, Maryland  20770
 *   
 *   Tel:    (301) 313-0053
 *   email:  info@trxsystems.com
 * 
 *****************************************************************************/
package com.trx.neon.api.neonEnvironment;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import com.trx.neon.api.neon.model.PositionConstraint;
import com.trx.neon.api.neonBeta.model.ElevationInfo;
import com.trx.neon.api.neonBeta.model.ManualConstraint;
import com.trx.neon.api.neonEnvironment.model.DownloadOptions;
import com.trx.neon.api.neonEnvironment.model.interfaces.INeonBuildingListener;
import com.trx.neon.api.neonEnvironment.model.interfaces.INeonFloorPlanListener;
import com.trx.neon.api.neonEnvironment.model.LatLongRect;
import com.trx.neon.api.neonEnvironment.model.NeonBuilding;
import com.trx.neon.api.neonEnvironment.model.NeonBuildingFloor;
import com.trx.neon.api.neon.Neon;

import java.util.UUID;

import static com.trx.neon.api._internal.NeonImpl.checkLocation;
import static com.trx.neon.api._internal.NeonImpl.neon;

/**
 * The main entry point for integrating Neon Environment Services into your application.
 * @author Dan
 */
public class NeonEnvironment extends Neon {

	/**
	 * Issues a request to download buildings in the specified area.  If they have previously been downloaded,
	 * it will return the cached result without hitting the network.
	 *
	 * @param bounds   The area defined by four latitude-longitude pairs
	 * @param callback Called on completion with status information (on main looper)
	 */
	public static void downloadBuildingsInArea(LatLongRect bounds, INeonBuildingListener callback) {
		checkLocation(bounds.Northeast.Latitude, bounds.Northeast.Longitude);
		checkLocation(bounds.Southwest.Latitude, bounds.Southwest.Longitude);
		neon.downloadBuildingsInArea(bounds, new Handler(Looper.getMainLooper()), callback, DownloadOptions.CACHED);
	}

	/**
	 * Issues a request to download buildings in the specified area.  If they have previously been downloaded,
	 * it will return the cached result without hitting the network.
	 *
	 * @param bounds   The area defined by four latitude-longitude pairs
	 * @param h        The handler, used for the status notification callback
	 * @param callback Called on completion with status information
	 */
	public static void downloadBuildingsInArea(LatLongRect bounds, Handler h, INeonBuildingListener callback) {
		checkLocation(bounds.Northeast.Latitude, bounds.Northeast.Longitude);
		checkLocation(bounds.Southwest.Latitude, bounds.Southwest.Longitude);
		neon.downloadBuildingsInArea(bounds, h, callback, DownloadOptions.CACHED);
	}

	/**
	 * Issues a request to download buildings in the specified area.  If they have previously been downloaded and the
	 * download options are set to CACHED, it will return the cached result without hitting the network.  If the options
	 * are set to FORCE_REFRESH it will re-download all buildings in the area.
	 *
	 * @param bounds          The area defined by four latitude-longitude pairs
	 * @param h               The handler, used for status notifications
	 * @param listener        Called on completion with a list of NeonBuildings and status
	 * @param downloadOptions If FORCE_REFRESH, will force the Neon Location Service to requery from the cloud.  If CACHED,
	 *                        calls to getBuildingsInArea may return cached values
	 */
	public static void downloadBuildingsInArea(LatLongRect bounds, Handler h, INeonBuildingListener listener, DownloadOptions downloadOptions) {
		checkLocation(bounds.Northeast.Latitude, bounds.Northeast.Longitude);
		checkLocation(bounds.Southwest.Latitude, bounds.Southwest.Longitude);
		neon.downloadBuildingsInArea(bounds, h, listener, downloadOptions);
	}

	/**
	 * retrieves a specific building by id that has already been downloaded to the Neon Location Service.
	 *
	 * @param id is the UUID identifier for a building
	 * @return A NeonBuilding or null if not found
	 */
	public static NeonBuilding getBuilding(UUID id) {
		return neon.getBuilding(id);
	}

	/**
	 * Downloads a floor plan image that corresponds with a NeonBuildingFloor.  This operation will
	 * download the floor plan if it has not already been cached in the Neon Location Service.  If the image
	 * can be retrieved it will decode a bitmap and return the results via the INeonFloorPlanListener.
	 * This is potentially a slow operation and work is done on the passed in handler - don't pass a handler
	 * to the main looper if you don't want to block the main thread.
	 *
	 * @param context         The context of the calling application
	 * @param floor           The NeonBuildingFloor object used to retrieve the floor plan image
	 * @param h               The handler, used for bitmap decoding and status notification. Work is performed on this handler.
	 * @param listener        Called on completion with status information and, if successful, a NeonFloorPlan
	 */
	public static void downloadFloorPlan(Context context, NeonBuildingFloor floor, Handler h, INeonFloorPlanListener listener)
	{
		neon.getFloorPlan(context, floor, h, listener);
	}

	public static void addBuildingConstraint(PositionConstraint constraint, UUID buildingID, int floor)
	{
		checkLocation(constraint.getLatitude(), constraint.getLongitude());
		neon.addManualConstraint(new ManualConstraint(constraint.getLatitude(), constraint.getLongitude(), constraint.getError(), ElevationInfo.OnFloor(buildingID, floor)));
	}
}
