package com.tenqube.visual_third.analysis;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.support.v4.content.LocalBroadcastManager;

import com.tenqube.visual_third.entity.Condition;
import com.tenqube.visual_third.entity.JoinedContent;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.analysis.Transaction;
import com.tenqube.visual_third.repository.AnalysisRepository;

import java.util.ArrayList;

import static com.tenqube.visual_third.analysis.AnalysisHelper.ascAnalysis;
import static com.tenqube.visual_third.analysis.AnalysisHelper.getContent;
import static com.tenqube.visual_third.analysis.AnalysisHelper.getValue;
import static com.tenqube.visual_third.parser.loader.OneLoader.ACTION_TRANSACTION_RECEIVED;
import static com.tenqube.visual_third.util.Utils.isEmpty;

public class AnalysisServiceImpl implements AnalysisService{

    private final Context context;
    private final AnalysisRepository repository;

    private ArrayList<Analysis> results = new ArrayList<>();

    private long time;
    private boolean isReceived;

    public AnalysisServiceImpl(Context context) {
        this.context = context;
        this.repository = AnalysisRepository.getInstance(context);
        IntentFilter filter = new IntentFilter();
        filter.addAction(ACTION_TRANSACTION_RECEIVED);
        LocalBroadcastManager.getInstance(context).registerReceiver(receiver, filter);
    }

    @Override
    public ArrayList<Analysis> loadAnalysisList(final ArrayList<Transaction> transactions) {

        FunctionMap functionMap = null;
        try {

            initCache();
            // 캐시가 존재하면 ? 바로 결과값 리턴
            if(!isEmpty(results)) return results;

            time = System.currentTimeMillis();

            functionMap = new FunctionMap();

            // 구한값을 가지고 계산이 필요한 부분 처리 해서 Funcmap 값 세팅
            functionMap.calculate(transactions);

            // 조건 체크하는 분석 정보 필터
            ArrayList<JoinedContent> contents = getSatisfiedContents(repository.loadContents());

            // 조건에 만족하는 분석 내용과 데이터 합치기
            mergeContents(contents);

            ascAnalysis(results);

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            FunctionMap.values.clear();
            FunctionMap.tranIds.clear();
            functionMap = null;
        }

        return results;
    }

    /**
     * 캐쉬 초기화 조건
     * 1. 내역 수신시
     * 2. 내역 편집시
     * 3. 60 초
     */
    private void initCache() {
        if(isReceived || isTimeOver()) {
            isReceived = false;
            time = 0;
            results.clear();
        }
    }

    private boolean isTimeOver() {
        return ((System.currentTimeMillis() - time) / 1000) > 60;
    }

    private LocalBroadCastReceiver receiver = new LocalBroadCastReceiver();

    public class LocalBroadCastReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {
            isReceived = true;
        }
    }

    private ArrayList<JoinedContent> getSatisfiedContents(ArrayList<JoinedContent> contents) {

        ArrayList<JoinedContent> results = new ArrayList<>();
        for(JoinedContent content : contents) {

            // 조건 체크
            boolean isSatisfied = false;
            ArrayList<Condition> conditions = content.getConditions();

            for(Condition condition : conditions) {
                isSatisfied = AnalysisHelper.isSatisfied(condition);
                if(!isSatisfied) break;
            }

            if(isSatisfied) {
                results.add(content);
            }
        }
        return results;
    }

    private void mergeContents(ArrayList<JoinedContent> contents) {

        for(JoinedContent content : contents) {

            try {
                String image = getValue(content.getContent().getImage());
                String label = getValue(content.getContent().getLabel());
                String lContent = getContent(content.getContent().getlContent(), content.getContent().getlKeys());
                String mContent = getContent(content.getContent().getmContent(), content.getContent().getmKeys());


                Analysis analysis = new Analysis(image,
                        label,
                        lContent,
                        mContent,
                        mergeTranIds(content.getContent().getRawKeys()),
                        content.getContent().getCategoryPriority()
                        );

                results.add(analysis);
            } catch (Exception e) {

            }

        }

    }

    private ArrayList<Integer> mergeTranIds(String rawKeys) {

        ArrayList<Integer> tranIds = new ArrayList<>();
        String[] keys = rawKeys.split(",");
        for(String key : keys) {
            ArrayList<Integer> tempTranIds = FunctionMap.tranIds.get(Integer.parseInt(key));
            if(tempTranIds != null)
                tranIds.addAll(tempTranIds);
        }

        return tranIds;

    }

}
