package com.tenqube.visual_third.ui;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.v4.app.FragmentManager;
import android.support.v4.widget.SwipeRefreshLayout;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;
import android.view.ViewTreeObserver;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Toast;

import com.tenqube.visual_third.BuildConfig;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.parser.loader.BulkLoader;
import com.tenqube.visual_third.repository.VisualRepository;

import org.json.JSONException;
import org.json.JSONObject;

import static com.tenqube.visual_third.Constants.CATEGORY_VERSION;
import static com.tenqube.visual_third.util.Utils.isNotiEnabled;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualWebActivity extends AppCompatActivity {

    public static final String ARG_PATH = "ARG_PATH";

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int NOTIFICATION_REQUEST_CODE = 10;

    public static final int FINISH = 0;
    public static final int PROGRESS = 1;
    public static final int SHOW_PROGRESS_BAR = 2;
    public static final int ERROR = 3;
    private ProgressDialogFragment progressDialog;
    private PrefManager prefManager;
    private VisualApi api;
    private VisualRepository repository;
    private boolean isPageLoaded;
    private boolean shouldRefresh;

    public static void startActivity(Context context, String path) throws AuthException {
        if(TextUtils.isEmpty(PrefManager.getInstance(context).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP | Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_CLEAR_TASK);
        context.startActivity(intent);
    }


    private String path;
    private WebView webview;
    private BulkLoader bulkLoader;
    private WebAppInterface webAppInterface;
    private SwipeRefreshLayout swipeRefreshLayout;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        try {
            api = VisualApi.getInstance(getApplicationContext());
            repository = VisualRepository.getInstance(getApplicationContext());
            prefManager = PrefManager.getInstance(getApplicationContext());
            path = getIntent().getStringExtra(ARG_PATH);

            setContentView(R.layout.activity_visual_web);

            swipeRefreshLayout = (SwipeRefreshLayout) findViewById(R.id.swipe_refresh_layout);
            swipeRefreshLayout.setEnabled(false);
            swipeRefreshLayout.setColorSchemeResources(
                    R.color.colorPopupRed,
                    R.color.colorPopupRed,
                    R.color.colorPopupRed);

            swipeRefreshLayout.getViewTreeObserver().addOnScrollChangedListener(new ViewTreeObserver.OnScrollChangedListener() {
                @Override
                public void onScrollChanged() {

                    LOGI("ONSCROLL", "scrolly" + webview.getScrollY(), mIsDebug);
                    if(webview != null && shouldRefresh) {
                        if (webview.getScrollY() == 0)
                            swipeRefreshLayout.setEnabled(true);
                        else
                            swipeRefreshLayout.setEnabled(false);
                    }

                }
            });

            swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    swipeRefreshLayout.setRefreshing(false);
                    webview.reload();
                }
            });

            webview = (WebView) findViewById(R.id.visual_web_view);

            webview.setWebViewClient(new WebViewClient() {
                @Override
                public void onPageFinished(WebView view, String url) {
                    super.onPageFinished(view, url);

                    if(BuildConfig.DEBUG)
                        Toast.makeText(getApplicationContext(), url, Toast.LENGTH_SHORT).show();
                }

            });
            webAppInterface = new WebAppInterface(this, webview, new BottomDialog(this),
                    VisualRepository.getInstance(getApplicationContext()), api);
            webview.addJavascriptInterface(webAppInterface, "visual");
            webview.getSettings().setJavaScriptEnabled(true);

            if(prefManager.loadStringValue(PrefManager.QUALIFIER, "dev").equals("dev"))
                webview.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);

            // 카테고리 동기화
            syncCategory();

            if(!prefManager.isEnabled(PrefManager.BULK_EXECUTED, false)) { // 벌크 파싱 진행

                LOGI(TAG, "벌크 파싱 진행", mIsDebug);
                startBulkParsing();

            } else {

                // 웹 가계부 호출
                loadUrl(path);

                // 검색 실패 내역 다시 보내기
                syncSearchFailedTransactions();
            }

        } catch (Exception e) {
            finish();
        }
    }

    public void onPagedLoaded(final boolean shouldRefresh) {
        this.shouldRefresh = shouldRefresh;

        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
                swipeRefreshLayout.setEnabled(shouldRefresh);
            }
        });
    }

    private void loadUrl(String path) {
        if(webview != null) {
            isPageLoaded = false;
            String url = getBaseUrl() + path;
            webview.loadUrl(url);
        }
    }

    private String getBaseUrl() {
        String url = prefManager.loadStringValue(PrefManager.WEB_URL, "");
        url = url.replace("https://", "");
        url = url.replace("http://", "");
        String version = BuildConfig.WEB_VERSION + ".";
        return "http://" + version + url + "/";
    }

    /**
     * 검색 실패 내역 다시 보내기
     */
    private void syncSearchFailedTransactions() {
        new Thread(new Runnable() {
            @Override
            public void run() {
                SearchCompanyRequest request = repository.loadFailedSearchTransactions();

                if(request != null) api.searchCompany(request);

                api.syncTransactions(null);
            }
        }).start();

    }

    /**
     * 카테고리 정보 동기화
     */
    private void syncCategory() {
        api.syncCategory(prefManager.loadIntValue(PrefManager.CATEGORY_VERSION, CATEGORY_VERSION));
    }


    private Handler handler = new Handler(new Handler.Callback() { // 핸들러로 전달받은 값 progress 값 세팅해주기
        @Override
        public boolean handleMessage(final Message msg) {

            switch (msg.what) {
                case PROGRESS:
                    if(progressDialog != null && progressDialog.isAdded()) {
                        Bundle bundle = msg.getData();
                        if(bundle != null) {
                            progressDialog.setProgressValue(bundle.getInt("now"), bundle.getInt("total"), bundle.getInt("percent"));
                        }
                    }
                    break;

                case SHOW_PROGRESS_BAR:

                    try {
                        if(progressDialog == null) {
                            progressDialog = ProgressDialogFragment.newInstance(VisualWebActivity.this);
                            progressDialog.setCancelable(false);
                        }

                        if(!progressDialog.isAdded()) {
                            FragmentManager fragmentManager = getSupportFragmentManager();
                            progressDialog.show(fragmentManager, ProgressDialogFragment.TAG);
                        }

                    } catch (Exception e) {
                        progressDialog = null;
                    }

                    break;
                case FINISH:
                    prefManager.saveBoolean(PrefManager.BULK_EXECUTED, true);
                    if(progressDialog != null && progressDialog.isAdded()) progressDialog.dismiss();
                    loadUrl(path);

                    break;
                default:
                    LOGI(TAG, "handler:" + msg.what, mIsDebug);
                    break;
            }

            return false;
        }
    });

    /**
     * 1. 퍼미션 체크
     * 2. 프로그레스 다이얼로그 show
     * 3. 지난 6개월 문자 파싱 시작
     */
    private void startBulkParsing() {

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {

                    LOGI(TAG, "startBulkParsing start", mIsDebug);
                    bulkLoader = new BulkLoader(VisualWebActivity.this, handler, new BulkLoader.OnSmsLoadFinished() {
                        @Override
                        public void onFinished(boolean isError) {
                            try {
                                if(handler != null) handler.sendEmptyMessage(FINISH);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
                    // 2. 프로그레스 다이얼로그 show
                    LOGI(TAG, "2. 프로그레스 다이얼로그 show", mIsDebug);
                    handler.sendEmptyMessage(SHOW_PROGRESS_BAR);

                    // 3. 지난 6개월 문자 파싱 시작
                    LOGI(TAG, "3. 지난 6개월 문자 파싱 시작", mIsDebug);
                    bulkLoader.doParsing();

                } catch (Exception e) {
                    handler.sendEmptyMessage(ERROR);
                }

            }
        }).start();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        switch (requestCode) {
            case NOTIFICATION_REQUEST_CODE:
                boolean isEnabled = isNotiEnabled(getApplicationContext());

                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("isNotiEnabled", isEnabled);
                    String json = jsonObject.toString();
                    if(webAppInterface != null) webAppInterface.onNotiCallback(json);

                } catch (JSONException e) {
                    e.printStackTrace();
                }
                break;

        }
    }

    @Override
    public void onBackPressed() {

        try {
            if(isPageLoaded && webAppInterface != null) {
                webAppInterface.onFinish();
            } else {
                super.onBackPressed();
            }
        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        overridePendingTransition(0, 0);
    }

    @Override
    public void onPause() {
        super.onPause();
        overridePendingTransition(0, 0);
    }

}
