package com.tenqube.visual_third;

import android.Manifest;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;

import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.Validator;

import java.util.Calendar;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.Constants.DEV;
import static com.tenqube.visual_third.Constants.PROD;
import static com.tenqube.visual_third.manager.PrefManager.UID;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private PrefManager prefManager;
    private AppCompatActivity activity;
    private VisualApi api;

    private ParserService parserService;

    public VisualServiceImpl(@NonNull AppCompatActivity activity, String apiKey, String qualifier) throws ParameterException {

        if(TextUtils.isEmpty(apiKey)) throw new ParameterException("apiKey is null");

        if(TextUtils.isEmpty(qualifier) || !(Constants.DEV.equals(qualifier) || Constants.PROD.equals(qualifier)))
            throw new ParameterException("check qualifier value (" + DEV + ", " + PROD + ")");

        this.activity = activity;

        VisualRepository.getInstance(activity);
        prefManager = PrefManager.getInstance(activity);
        prefManager.saveStringValue(PrefManager.API_KEY, apiKey);
        prefManager.saveStringValue(PrefManager.QUALIFIER, qualifier);
        api = VisualApi.getInstance(activity);
        parserService = ParserService.getInstance(activity);
    }

    @Override
    public void signUp(@NonNull final String custId, @NonNull final OnResultListener onResultListener) throws ParameterException, AuthException {

        Validator.isStr(custId, 500);
        if(!TextUtils.isEmpty(prefManager.loadStringValue(UID, ""))) throw new AuthException("already joined.");

        new Thread(new Runnable() {
            @Override
            public void run() {

                LOGI(TAG, "가입", mIsDebug);
                SignUpResponse response = null;
                try {
                    response = api.signUp(new SignUpRequest(custId));
                } catch (ParameterException e) {
                    e.printStackTrace();
                }

                if (response == null) {
                    onResultListener.onSuccess(false, "network error");

                } else {

                    // login request succeed, new token generated
                    prefManager.saveStringValue(PrefManager.SEARCH_URL, response.getResults().getSearch().getUrl());
                    prefManager.saveStringValue(PrefManager.SEARCH_API_KEY, response.getResults().getSearch().getApiKey());
                    prefManager.saveStringValue(UID, response.getResults().getAuthorization().getSdk());
                    prefManager.saveLongValue(PrefManager.SIGN_UP_TIME, Calendar.getInstance().getTimeInMillis());
                    if (response.getResults().getWeb() != null)
                        prefManager.saveStringValue(PrefManager.WEB_URL, response.getResults().getWeb().getUrl());

                    onResultListener.onSuccess(true, "");
                }
            }
        }).start();
    }

    @Override
    public void startVisual(String custId) throws ParameterException, SecurityException, AuthException {

        LOGI(TAG, "웹 가계부 호출", mIsDebug);

        if (!prefManager.isEnabled(PrefManager.BULK_EXECUTED, false) &&
                ActivityCompat.checkSelfPermission(activity, Manifest.permission.READ_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_MMS) != PackageManager.PERMISSION_GRANTED) {

            throw new SecurityException("Please grant permissions " +
                    "(Manifest.permission.READ_SMS, " +
                    "Manifest.permission.RECEIVE_SMS, " +
                    "Manifest.permission.RECEIVE_MMS)");
        }

        // 가입안된 경우
        if(TextUtils.isEmpty(prefManager.loadStringValue(PrefManager.UID, ""))) {
            signUp(custId, new OnResultListener() {
                @Override
                public void onSuccess(boolean isSuccess, String msg) {
                    if(isSuccess) {
                        activity.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                try {
                                    VisualWebActivity.startActivity(activity, "");
                                } catch (AuthException e) {
                                    e.printStackTrace();
                                }
                            }
                        });
                    }
                }
            });
        } else {
            VisualWebActivity.startActivity(activity, "");
        }

    }

    @Override
    public void setDebugMode(boolean isDebug) {
        LOGI(TAG, "setDebugMode", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.setDebugMode(isDebug);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.initDb();
        }

    }

    private boolean isActive() {
        return activity != null && !activity.isFinishing();
    }
}
