package com.tenqube.visual_third.sms.data.source.smsrule

import android.content.Context
import android.util.Log
import com.google.gson.Gson
import com.tenqube.visual_third.manager.PrefManager
import com.tenqube.visual_third.sms.data.api.RuleService
import com.tenqube.visual_third.sms.data.api.SmsRulesDto
import com.tenqube.visual_third.sms.data.api.dto.NotificationDto
import com.tenqube.visual_third.sms.data.api.dto.SmsRuleDto
import com.tenqube.visual_third.sms.data.db.dao.SmsRuleDao
import com.tenqube.visual_third.sms.data.db.model.SmsRuleModel
import com.tenqube.visual_third.sms.domain.dto.SMSResponse
import com.tenqube.visual_third.sms.domain.smsrule.SmsRule
import com.tenqube.visual_third.sms.domain.smsrule.SmsRuleRepository
import com.tenqube.visual_third.sms.domain.utils.Result
import com.tenqube.visual_third.sms.domain.utils.loadJSONFromAsset
import com.tenqube.visual_third.util.Utils
import java.util.*


class SmsRuleRepositoryImpl(private val context: Context,
                            private val smsRuleDao: SmsRuleDao,
                            private val ruleService: RuleService,
                            private val prefManager: PrefManager) : SmsRuleRepository {

    override suspend fun isActive(): Boolean {

        var isActive = prefManager.isEnabled(PrefManager.NOTI_PARSER_IS_ACTIVE, true)
        val diff = System.currentTimeMillis() - prefManager.loadLongValue(PrefManager.NOTI_PARSER_IS_ACTIVE_SYNC_TIME, 0)

        if(diff >= 24 * 60 * 60 * 1000) {
            isActive = ruleService.isActive()
            prefManager.saveBoolean(PrefManager.NOTI_PARSER_IS_ACTIVE, isActive)
        }

        return isActive
    }

    override suspend fun findBySender(sender: String): Result<List<SmsRule>> {

        return Result.Success(getRules()
                .filter {

                    it.sender.contains(sender) || sender.contains(it.sender) }.map {

                    SmsRule(it.id, it.sender, it.title, it.regExp, it.body)
        })
    }

    private suspend fun getRules(): List<SmsRule> {

        var localRules = smsRuleDao.findAll()

        if(localRules.isEmpty() || isOneDayAfter()) {
            val rules = ruleService.fetchSmsRules()

            rules.forEach {
                smsRuleDao.save(SmsRuleModel(0, it.id, it.sender, it.title, it.regExp, it.body))
            }
            localRules = smsRuleDao.findAll()


            if(localRules.isEmpty()) {
                localRules = getRulesFromAsset().map {
                    SmsRuleModel(0, it.id, it.sender, it.title, it.regExp, it.body)
                }
            }
        }


        return localRules.map {
            it.toEntity()
        }

    }

    private fun isOneDayAfter(): Boolean {
        val calendar = Calendar.getInstance()
        val beforeDay = prefManager.loadStringValue(PrefManager.ONE_DAY, "")
        val today = Utils.toYmdFormat(calendar)
        prefManager.saveStringValue(PrefManager.ONE_DAY, today)
        return beforeDay != today
    }

    private fun getRulesFromAsset(): List<SmsRuleDto> {

        val rulesJson = "sms_rule.json".loadJSONFromAsset(context.assets)
        val json = Gson().fromJson(rulesJson, SmsRulesDto::class.java)
        return json.rules
    }

    override suspend fun saveNotification(result: SMSResponse) {
        ruleService.saveNotification(NotificationDto(result.id, result.date, result.body))
    }


}