package com.tenqube.visual_third.sms.data.api.util

import android.util.Log
import com.google.gson.annotations.SerializedName
import com.tenqube.visual_third.util.Utils.fromJson
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import retrofit2.HttpException
import timber.log.Timber
import java.io.IOException
import java.io.Serializable


data class ErrorResponse(@SerializedName("code") val code: String?,
                         @SerializedName("name") val name: String?,
                         @SerializedName("message") val message: String) : Serializable


sealed class ResultWrapper<out T> {
    data class Success<out T>(val value: T): ResultWrapper<T>()
    data class GenericError(val code: Int? = null, val error: ErrorResponse? = null): ResultWrapper<Nothing>()
    object NetworkError: ResultWrapper<Nothing>()

}

suspend fun <T> safeApiCall(
        dispatcher: CoroutineDispatcher,
        apiCall: suspend () -> T
): ResultWrapper<T> {

    return withContext(dispatcher) {
        try {
            ResultWrapper.Success(apiCall.invoke())
        } catch (throwable: Throwable) {

            Log.i("parsing", "throwable" + throwable.toString())

            when (throwable) {
                is IOException -> ResultWrapper.NetworkError
                is HttpException -> {
                    val code = throwable.code()

                    val errorResponse = convertErrorBody(throwable)
                    ResultWrapper.GenericError(code, errorResponse)
                }
                else -> {
                    ResultWrapper.GenericError(null, null)
                }
            }
        }
    }
}

private fun convertErrorBody(throwable: HttpException): ErrorResponse? {
    return try {
        val result = throwable.response()?.errorBody()?.string()
        result?.let {
            return fromJson(
                    result,
                    ErrorResponse::class.java
            )
        }
        return null

    } catch (exception: Exception) {
        null
    }
}