package com.tenqube.visual_third.sms.data.api

import android.content.Context
import android.util.Log
import com.google.gson.Gson
import com.google.gson.annotations.SerializedName
import com.tenqube.visual_third.manager.PrefManager
import com.tenqube.visual_third.sms.data.api.dto.NotificationDto
import com.tenqube.visual_third.sms.data.api.dto.SmsRuleDto
import com.tenqube.visual_third.sms.data.api.util.ResultWrapper
import com.tenqube.visual_third.sms.data.api.util.safeApiCall
import com.tenqube.visual_third.sms.domain.utils.Constants
import com.tenqube.visual_third.sms.domain.utils.loadJSONFromAsset
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import timber.log.Timber
import java.io.IOException
import java.lang.Exception

data class SmsRulesDto(@SerializedName("rules") val rules: List<SmsRuleDto>)

class RuleServiceImpl(private val api: NotiParserApi,
                      private val prefManager: PrefManager,
                      private val ioDispatcher: CoroutineDispatcher = Dispatchers.IO): RuleService {

    private fun getUrl(path: String): String {
        return if(prefManager.loadStringValue(PrefManager.QUALIFIER, com.tenqube.visual_third.Constants.DEV) == com.tenqube.visual_third.Constants.DEV) {
            NotiParserApi.DEV_URL + path
        } else {
            NotiParserApi.PROD_URL + path
        }
    }

    @Throws(IOException::class)
    private fun getHeader(): Map<String, String> {
        val map = HashMap<String, String>()

        val custId = prefManager.loadStringValue(PrefManager.CUST_ID, "")
        if(custId.isEmpty()) {
            throw IOException("custId is null")
        }
        map["serviceType"] = "clip"
        map["clientId"] = "custId"

        map["x-api-key"] = if(prefManager.loadStringValue(PrefManager.QUALIFIER, com.tenqube.visual_third.Constants.DEV) == com.tenqube.visual_third.Constants.DEV)
            "wEkaXnTgci1mwzm7VuJoi6TL6grPyMuf4PKxgmG1" else "tay2bGJMIvN3e2TvUf9YBcHSv9CrNa6aUjmMkyc0"
        return map

    }

    override suspend fun isActive(): Boolean  = withContext(ioDispatcher){

        return@withContext try {
            when (val response = safeApiCall(ioDispatcher) {
                api.getStatus(getUrl(NotiParserApi.STATUS), getHeader())
            }) {
                is ResultWrapper.GenericError -> {
                    response.code != 403
                }
                else -> {
                    true
                }
            }
        } catch (e: Exception) {
            true
        }
    }

    override suspend fun fetchSmsRules(): List<SmsRuleDto> = withContext(ioDispatcher){

        return@withContext try {
             when (val response = safeApiCall(ioDispatcher) {
                api.fetchSmsRules(getUrl(NotiParserApi.SMS_RULE), getHeader())
            }) {
                is ResultWrapper.Success -> {
                    response.value.rules
                } else -> {
                    emptyList()
                }
            }
        } catch (e: Exception) {
            listOf<SmsRuleDto>()
        }
    }

    override suspend fun saveNotification(notificationDto: NotificationDto) {

        withContext(ioDispatcher) {
            try {
                when (val response = safeApiCall(ioDispatcher) {
                    api.saveNotification(getUrl(NotiParserApi.NOTIFICATION), getHeader(), notificationDto)
                }) {
                    is ResultWrapper.Success -> {

                        Log.i("parsing", "saveNotification success")

                    }

                    is ResultWrapper.NetworkError -> {
                        Log.i("parsing", "saveNotification NetworkError")

                    }

                    is ResultWrapper.GenericError -> {
                        Log.i("parsing", "saveNotification GenericError code: " + response.code)

                    }

                    else -> {
                        Log.i("parsing", "saveNotification UnknownError")

                    }

                }
            } catch (e: Exception) {

            }
        }


    }
}