package com.tenqube.visual_third.sms

import android.content.Context
import com.jakewharton.retrofit2.adapter.kotlin.coroutines.CoroutineCallAdapterFactory
import com.tenqube.visual_third.manager.PrefManager
import com.tenqube.visual_third.sms.data.api.NotiParserApi
import com.tenqube.visual_third.sms.data.api.RuleService
import com.tenqube.visual_third.sms.data.db.SmsDatabase
import com.tenqube.visual_third.sms.data.db.dao.SmsDao
import com.tenqube.visual_third.sms.data.db.dao.SmsRuleDao
import com.tenqube.visual_third.sms.data.source.sms.SmsRepositoryImpl
import com.tenqube.visual_third.sms.data.source.smsrule.SmsRuleRepositoryImpl
import com.tenqube.visual_third.sms.domain.sms.DeleteAllSmsUseCase
import com.tenqube.visual_third.sms.domain.sms.DeleteSmsByDateUseCase
import com.tenqube.visual_third.sms.domain.sms.SmsRepository
import com.tenqube.visual_third.sms.domain.smsrule.ParseSMSByRuleUseCase
import com.tenqube.visual_third.sms.domain.smsrule.SmsRuleRepository
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit

object Injection {

    fun provideOkHttpClient(prefManager: PrefManager): OkHttpClient {
        return OkHttpClient.Builder().apply {
            this.connectTimeout(5, TimeUnit.SECONDS)
            this.readTimeout(5, TimeUnit.SECONDS)
            this.writeTimeout(5, TimeUnit.SECONDS)
            val logInter = HttpLoggingInterceptor()
            logInter.level = HttpLoggingInterceptor.Level.BODY
            this.addInterceptor(logInter)
        }.build()
    }

    fun provideRetrofit(client: OkHttpClient): Retrofit {
        return Retrofit.Builder()
                .client(client)
                .baseUrl(NotiParserApi.PROD_URL)
                .addConverterFactory(GsonConverterFactory.create())
                .addCallAdapterFactory(CoroutineCallAdapterFactory())
                .build()
    }

    fun provideNotiParserApi(retrofit: Retrofit): NotiParserApi {
        return retrofit.create(NotiParserApi::class.java)
    }

    fun provideDb(context: Context): SmsDatabase {
        return SmsDatabase.get(context.applicationContext)
    }

    fun provideSmsDao(db: SmsDatabase): SmsDao {
        return db.smsDao()
    }

    fun provideSmsRuleDao(db: SmsDatabase): SmsRuleDao {
        return db.smsRuleDao()
    }

    fun provideSmsRepository(smsDao: SmsDao): SmsRepository {
        return SmsRepositoryImpl(smsDao)
    }

    fun provideSmsRuleRepository(context: Context, smsRuleDao: SmsRuleDao, ruleService: RuleService, prefManager: PrefManager): SmsRuleRepository {
        return SmsRuleRepositoryImpl(context, smsRuleDao, ruleService, prefManager)
    }

    fun provideDeleteAllSmsUseCase(smsRepository: SmsRepository): DeleteAllSmsUseCase {
        return DeleteAllSmsUseCase(smsRepository)
    }

    fun provideDeleteSmsByDateUseCase(smsRepository: SmsRepository): DeleteSmsByDateUseCase {
        return DeleteSmsByDateUseCase(smsRepository)
    }

    fun provideParseSMSByRuleUseCase(smsRuleRepository: SmsRuleRepository, smsRepository: SmsRepository): ParseSMSByRuleUseCase {
        return ParseSMSByRuleUseCase(smsRuleRepository, smsRepository)
    }
}