package com.tenqube.visual_third.ui;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Color;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.bottomsheet.BottomSheetBehavior;
import com.google.android.material.bottomsheet.BottomSheetDialog;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.model.js.SelectBoxInfo;
import com.tenqube.visual_third.model.js.SelectBoxRequest;

import java.util.ArrayList;

import static com.tenqube.visual_third.util.Utils.changeColor;
import static com.tenqube.visual_third.util.Utils.dpToPx;

public class BottomDialog {

    private  OnBottomListener bottomListener;
    private final Context activity;

    private BottomSheetDialog bottomSheetDialog;
    private int peekHeight;
    public interface OnBottomListener {
        void onItemSelected(SelectBoxInfo selectBoxInfo);
    }

    public BottomDialog(Context activity) {
        this.activity = activity;
    }

    public void setBottomListener(OnBottomListener bottomListener) {
        this.bottomListener = bottomListener;
        peekHeight = Resources.getSystem().getDisplayMetrics().heightPixels - dpToPx(24);
    }

    public void showBottomDialog(SelectBoxRequest selectBoxRequest) {

        try {
            if(selectBoxRequest == null) return;

            if (bottomSheetDialog == null) {
                bottomSheetDialog = new BottomSheetDialog(activity);
                bottomSheetDialog.getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_PAN);
            }

            if (!isShowing()) {
                View view = LayoutInflater.from(activity).inflate(R.layout.bottom_sheet_view, null);
                if (view != null) {

                    TextView titleTextView = (TextView) view.findViewById(R.id.title);
                    titleTextView.setText(selectBoxRequest.getTitle());

                    view.findViewById(R.id.close).setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View view) {
                            dismiss();
                        }
                    });

                    RecyclerView recyclerView = (RecyclerView) view.findViewById(R.id.recyclerView);

                    if(selectBoxRequest.getData().size() > 3) {
                        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
                        params.height = peekHeight;
                        recyclerView.setLayoutParams(params);
                    }
                    final LinearLayoutManager linearLayoutManager = new LinearLayoutManager(activity);
                    recyclerView.setLayoutManager(linearLayoutManager);
                    recyclerView.setHasFixedSize(true);
                    recyclerView.setAdapter(new BottomAdapter(selectBoxRequest, bottomListener));

                    bottomSheetDialog.setContentView(view);

                    BottomSheetBehavior mBehavior = BottomSheetBehavior.from((View) view.getParent());
                    mBehavior.setPeekHeight(peekHeight);

                    bottomSheetDialog.show();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void dismiss() {
        if(isShowing()) {
            bottomSheetDialog.dismiss();
        }
    }

    private boolean isShowing() {
        return bottomSheetDialog != null && bottomSheetDialog.isShowing();
    }

    interface OnClickListener {
        void onClick(int position);
    }

    public static class BottomAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> implements  OnClickListener{


        private final ArrayList<SelectBoxInfo> selectBoxInfos;
        private final OnBottomListener bottomListener;
        private int selectedPos = -1;
        private int selectedColor;

        BottomAdapter(SelectBoxRequest selectBoxRequests, OnBottomListener bottomListener) {
            this.selectBoxInfos = selectBoxRequests.getData();
            this.bottomListener = bottomListener;

            setSelectedColor(selectBoxRequests.getSelectedColor());
            setSelectedPos();
        }

        private void setSelectedColor(String color) {
            try {
                selectedColor = Color.parseColor(color);
            } catch (Exception e) {
                selectedColor = Color.parseColor("#ff051d");
            }
        }

        private void setSelectedPos() {

            int size = selectBoxInfos.size();
            for(int i = 0 ; i < size  ; i ++) {
                if(selectBoxInfos.get(i).isSelected()) {
                    selectedPos = i;
                    return;
                }
            }
        }

        @NonNull
        @Override
        public RecyclerView.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {

            ViewGroup view = (ViewGroup) LayoutInflater.from(parent.getContext()).inflate(R.layout.adapter_select_box, parent, false);
            return new BottomViewHolder(view, this);
        }

        @Override
        public void onBindViewHolder(@NonNull RecyclerView.ViewHolder holder, int position) {

            try {
                SelectBoxInfo selectBoxInfo = selectBoxInfos.get(position);
                selectBoxInfo.setSelected(selectedPos == position);
                BottomViewHolder bHolder = (BottomViewHolder)holder;
                bHolder.onBind(selectBoxInfo, selectedColor);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        @Override
        public int getItemCount() {
            return selectBoxInfos == null ? 0 : selectBoxInfos.size();
        }

        @Override
        public void onClick(int position) {
            this.selectedPos = position;
            notifyItemChanged(position);
            if(bottomListener != null) bottomListener.onItemSelected(selectBoxInfos.get(position));
        }
    }

    static class BottomViewHolder extends RecyclerView.ViewHolder {

        LinearLayout container;
        TextView nameTextView;

        BottomViewHolder(View convertView, final OnClickListener onClickListener) {
            super(convertView);

            convertView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    if(onClickListener != null && getAdapterPosition() != RecyclerView.NO_POSITION) onClickListener.onClick(getAdapterPosition());
                }
            });

            container = (LinearLayout) convertView.findViewById(R.id.container);

            nameTextView = (TextView) convertView.findViewById(R.id.name);

        }

        void onBind(SelectBoxInfo selectBoxInfo, int selectedColor) {
            if(selectBoxInfo.isSelected()) {
                changeColor(container, selectedColor);
                nameTextView.setTextColor(selectedColor);
            } else {
                changeColor(container, ContextCompat.getColor(nameTextView.getContext(), R.color.colorPopupGrey));
                nameTextView.setTextColor(nameTextView.getContext().getResources().getColor(android.R.color.black));
            }
            nameTextView.setText(selectBoxInfo.getName());
        }


    }
}
