package com.tenqube.visual_third.core;

import android.Manifest;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;

import com.tenqube.visual_third.OnResultListener;
import com.tenqube.visual_third.VisualService;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.util.Validator;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.Constants.DEV;
import static com.tenqube.visual_third.Constants.PROD;
import static com.tenqube.visual_third.manager.PrefManager.UID;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();



    private PrefManager prefManager;
    private AppCompatActivity activity;
    private VisualApi api;

    private ParserService parserService;

    public VisualServiceImpl(@NonNull AppCompatActivity activity) throws ParameterException {

        if(activity == null) throw new ParameterException("activity is null");

        this.activity = activity;
        prefManager = PrefManager.getInstance(activity);
        api = VisualApi.getInstance(activity);
        parserService = ParserService.getInstance(activity);

    }



    @Override
    public void setQualifier(String qualifier) throws ParameterException {

        if(DEV.equals(qualifier) || PROD.equals(qualifier)) {
            prefManager.saveStringValue(PrefManager.LAYER, qualifier);
        } else {
            throw new ParameterException("check qualifier value (" + DEV + ", " + PROD + ")");
        }

    }

    @Override
    public void signUp(@NonNull String custId, @NonNull OnResultListener onResultListener) throws ParameterException, AuthException {
        Validator.isCustId(custId);

        if(!TextUtils.isEmpty(prefManager.loadStringValue(UID, ""))) throw new AuthException("already joined.");

        LOGI(TAG, "가입", mIsDebug);
        api.signUp(new SignUpRequest(custId), onResultListener);
    }

    @Override
    public void startVisual() throws AuthException, SecurityException {
        if(TextUtils.isEmpty(prefManager.loadStringValue(UID, "")))
            throw new AuthException("UID is empty");

        if (!prefManager.isEnabled(PrefManager.BULK_EXECUTED, false) &&
                ActivityCompat.checkSelfPermission(activity, Manifest.permission.READ_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_MMS) != PackageManager.PERMISSION_GRANTED) {

            throw new SecurityException("Please grant permissions " +
                    "(Manifest.permission.READ_SMS, " +
                    "Manifest.permission.RECEIVE_SMS, " +
                    "Manifest.permission.RECEIVE_MMS)");
        }

        LOGI(TAG, "웹 가계부 호출", mIsDebug);
        VisualWebActivity.startActivity(activity, "");
    }



    @Override
    public void setDebugMode(boolean isDebug) {
        LOGI(TAG, "setDebugMode", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.setDebugMode(isDebug);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.initDb();
        }

    }

    private boolean isActive() {
        return activity != null && !activity.isFinishing();
    }
}
