package com.tenqube.visual_third.core;

import android.Manifest;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.app.FragmentManager;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;

import com.tenqube.visual_third.OnResultListener;
import com.tenqube.visual_third.VisualService;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.ui.ProgressDialogFragment;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.Validator;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.Constants.DEV;
import static com.tenqube.visual_third.Constants.PROD;
import static com.tenqube.visual_third.manager.PrefManager.UID;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int FINISH = 0;
    public static final int PROGRESS = 1;
    public static final int SHOW_PROGRESS_BAR = 2;
    public static final int ERROR = 3;

    private PrefManager prefManager;
    private AppCompatActivity activity;
    private VisualApi api;
    private ProgressDialogFragment progressDialog;
    private BulkLoader bulkLoader;
    private ParserService parserService;

    public VisualServiceImpl(@NonNull AppCompatActivity activity) throws ParameterException {

        if(activity == null) throw new ParameterException("activity is null");

        this.activity = activity;

        prefManager = PrefManager.getInstance(activity);
        api = VisualApi.getInstance(activity);
        parserService = ParserService.getInstance(activity);

        bulkLoader = new BulkLoader(activity, handler, new BulkLoader.OnSmsLoadFinished() {
            @Override
            public void onFinished(boolean isError) {
                try {
                    if(handler != null) handler.sendEmptyMessage(FINISH);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    private Handler handler = new Handler(new Handler.Callback() { // 핸들러로 전달받은 값 progress 값 세팅해주기
        @Override
        public boolean handleMessage(final Message msg) {

            if(isActive()) {
                switch (msg.what) {
                    case PROGRESS:
                        if(progressDialog != null && progressDialog.isAdded()) {
                            Bundle bundle = msg.getData();
                            if(bundle != null) {
                                progressDialog.setProgressValue(bundle.getInt("now"), bundle.getInt("total"), bundle.getInt("percent"));
                            }
                        }
                        break;

                    case SHOW_PROGRESS_BAR:

                        if(progressDialog == null) {
                            progressDialog = ProgressDialogFragment.newInstance();
                            progressDialog.setCancelable(false);
                        }

                        try {
                            if(!progressDialog.isAdded()) {
                                FragmentManager fragmentManager = (activity).getSupportFragmentManager();
                                progressDialog.show(fragmentManager, ProgressDialogFragment.TAG);
                            }
                        } catch (ClassCastException e) {
                            e.printStackTrace();
                        }

                        break;
                    default:
                        prefManager.saveBoolean(PrefManager.BULK_EXECUTED, true);
                        if(progressDialog != null && progressDialog.isAdded()) progressDialog.dismiss();
                        break;
                }
            }


            return false;
        }
    });

    @Override
    public void setQualifier(String qualifier) throws ParameterException {

        if(DEV.equals(qualifier) || PROD.equals(qualifier)) {
            prefManager.saveStringValue(PrefManager.LAYER, qualifier);
        } else {
            throw new ParameterException("check qualifier value (" + DEV + ", " + PROD + ")");
        }

    }

    @Override
    public void signUp(@NonNull String custId, @NonNull OnResultListener onResultListener) throws ParameterException, AuthException {
        Validator.isCustId(custId);

        if(!TextUtils.isEmpty(prefManager.loadStringValue(UID, ""))) throw new AuthException("already joined.");

        LOGI(TAG, "가입", mIsDebug);
        api.signUp(new SignUpRequest(custId), onResultListener);
    }

    @Override
    public void startVisual() throws AuthException, SecurityException {
        if(TextUtils.isEmpty(prefManager.loadStringValue(UID, "")))
            throw new AuthException("UID is empty");

        if(!prefManager.isEnabled(PrefManager.BULK_EXECUTED, false)) { // 벌크 파싱 진행
            LOGI(TAG, "벌크 파싱 진행", mIsDebug);
            startBulkParsing();
        } else { // 웹 가계부 호출
            LOGI(TAG, "웹 가계부 호출", mIsDebug);
            VisualWebActivity.startActivity(activity, "");
        }
    }

    /**
     * 1. 퍼미션 체크
     * 2. 프로그레스 다이얼로그 show
     * 3. 지난 6개월 문자 파싱 시작
     */
    private void startBulkParsing() throws SecurityException {

        LOGI(TAG, "startBulkParsing start", mIsDebug);

        if(isActive()) {

            // 1. 퍼미션 체크
            LOGI(TAG, "1. 퍼미션 체크", mIsDebug);

            if (ActivityCompat.checkSelfPermission(activity, Manifest.permission.READ_SMS) != PackageManager.PERMISSION_GRANTED &&
                    ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_SMS) != PackageManager.PERMISSION_GRANTED &&
                    ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_MMS) != PackageManager.PERMISSION_GRANTED) {

                throw new SecurityException("Please grant permissions " +
                        "(Manifest.permission.READ_SMS, " +
                        "Manifest.permission.RECEIVE_SMS, " +
                        "Manifest.permission.RECEIVE_MMS)");
            }

            new Thread(new Runnable() {
                @Override
                public void run() {
                    try {

                        // 2. 프로그레스 다이얼로그 show
                        LOGI(TAG, "2. 프로그레스 다이얼로그 show", mIsDebug);
                        handler.sendEmptyMessage(SHOW_PROGRESS_BAR);

                        // 3. 지난 6개월 문자 파싱 시작
                        LOGI(TAG, "3. 지난 6개월 문자 파싱 시작", mIsDebug);
                        bulkLoader.doParsing();

                    } catch (Exception e) {
                        handler.sendEmptyMessage(ERROR);
                    }

                }
            }).start();
        }

    }

    @Override
    public void setDebugMode(boolean isDebug) {
        LOGI(TAG, "setDebugMode", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.setDebugMode(isDebug);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.initDb();
        }

    }

    private boolean isActive() {
        return activity != null && !activity.isFinishing();
    }
}
