package com.tenqube.visual_third.core;

import android.Manifest;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.app.FragmentManager;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;

import com.tenqube.visual_third.BuildConfig;
import com.tenqube.visual_third.VisualService;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.SignUpInfo;
import com.tenqube.visual_third.ui.ProgressDialogFragment;
import com.tenqube.visual_third.util.Validator;

import tenqube.parser.core.ParserService;

import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int SHOW_PROGRESS_BAR = 2000;
    public static final int ERROR = 404;


    private static final String DEV_WEB_URL = "http://dev.kt-view.tenqube.kr/";
    private static final String PROD_WEB_URL = "http://prod.kt-view.tenqube.kr/";


    private PrefManager prefManager;
    private AppCompatActivity activity;
    private VisualApi api;

    private ProgressDialogFragment progressDialog;

    private BulkLoader bulkLoader;


    public VisualServiceImpl(@NonNull AppCompatActivity activity) throws ParameterException {

        if(activity == null) throw new ParameterException("activity is null");

        this.activity = activity;

        prefManager = PrefManager.getInstance(activity);
        api = VisualApi.getInstance(activity);

        bulkLoader = new BulkLoader(activity, handler, new BulkLoader.OnSmsLoadFinished() {
            @Override
            public void onFinished(boolean isError) {
                try {
                    if(handler != null) handler.sendEmptyMessage(404);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    private Handler handler = new Handler(new Handler.Callback() { // 핸들러로 전달받은 값 progress 값 세팅해주기
        @Override
        public boolean handleMessage(final Message msg) {

            if(isActive()) {
                switch (msg.what) {

                    case 1:
                        if(progressDialog != null && progressDialog.isAdded()) {
                            Bundle bundle = msg.getData();
                            if(bundle != null) {
                                progressDialog.setProgressValue(bundle.getInt("now"), bundle.getInt("total"), bundle.getInt("percent"));
                            }
                        }

                        break;

                    case SHOW_PROGRESS_BAR:

                        if(progressDialog == null) {
                            progressDialog = ProgressDialogFragment.newInstance();
                            progressDialog.setCancelable(false);
                        }

                        try {
                            if(!progressDialog.isAdded()) {
                                FragmentManager fragmentManager = (activity).getSupportFragmentManager();
                                progressDialog.show(fragmentManager, ProgressDialogFragment.TAG);
                            }

                        } catch (ClassCastException e) {
                            e.printStackTrace();
                        }

                        break;
                    case ERROR:
                        if(progressDialog != null && progressDialog.isAdded()) progressDialog.dismiss();
                        break;
                    default:
                        break;
                }
            }


            return false;
        }
    });

    /**
     * 1. validation 체크
     * 2. 퍼미션 체크
     * 3. 프로그레스 다이얼로그 show
     * 4. 가입
     * 5. 지난 6개월 문자 파싱 시작
     * 6. 프로그레스 다이얼로그 dismiss
     */
    @Override
    public void startBulkParsing(final String custId) throws ParameterException, SecurityException {

        LOGI(TAG, "startBulkParsing start", mIsDebug);

        if(isActive()) {
            // 1. validation 체크
            Validator.isCustId(custId);

            // 2. 퍼미션 체크
            if (ActivityCompat.checkSelfPermission(activity, Manifest.permission.READ_SMS) != PackageManager.PERMISSION_GRANTED &&
                    ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_SMS) != PackageManager.PERMISSION_GRANTED &&
                    ActivityCompat.checkSelfPermission(activity, Manifest.permission.RECEIVE_MMS) != PackageManager.PERMISSION_GRANTED) {

                throw new SecurityException("Please grant permissions " +
                        "(Manifest.permission.READ_SMS, " +
                        "Manifest.permission.RECEIVE_SMS, " +
                        "Manifest.permission.RECEIVE_MMS)");

            }


            new Thread(new Runnable() {
                @Override
                public void run() {
                    try {

                        // 3. 프로그레스 다이얼로그 show
                        LOGI(TAG, "3. 프로그레스 다이얼로그 show", mIsDebug);
                        handler.sendEmptyMessage(SHOW_PROGRESS_BAR);

                        // 4. 가입 동기형태로 구성
                        LOGI(TAG, "4. 가입 동기형태로 구성", mIsDebug);
                        api.signUp(new SignUpInfo(custId));

                        // 5. 지난 6개월 문자 파싱 시작
                        LOGI(TAG, "5. 지난 6개월 문자 파싱 시작", mIsDebug);
                        bulkLoader.doParsing();

                    } catch (Exception e) {
                        handler.sendEmptyMessage(ERROR);
                    }

                }
            }).start();
        }



    }



    /**
     * 웹 가계부 연결 URL
     * @return URL 정보
     */
    @Override
    public String loadUrl() throws AuthException {

        String accessToken = prefManager.loadStringValue(PrefManager.ACCESS_TOKEN, "");
        if(TextUtils.isEmpty(accessToken))
            throw new AuthException("accessToken is empty");

        String baseUrl = BuildConfig.DEBUG ? DEV_WEB_URL : PROD_WEB_URL;
        String tokenParam = "?authToken=";
        return baseUrl + tokenParam + accessToken;
    }



    @Override
    public void setDebugMode(boolean isDebug) {
        LOGI(TAG, "setDebugMode", mIsDebug);

        if(isActive()) {
            ParserService.getInstance(activity).setDebugMode(isDebug);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive()) {
            ParserService.getInstance(activity).initDb();
        }

    }

    private boolean isActive() {
        return activity != null && !activity.isFinishing();
    }
}
