package com.tenqube.visual_third.core;

import android.content.Context;

import com.google.gson.Gson;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.SignUpInfo;
import com.tenqube.visual_third.model.TranCompany;
import com.tenqube.visual_third.model.VisualToken;
import com.tenqube.visual_third.model.VisualTransaction;

import java.io.IOException;
import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import tenqube.parser.model.ParsingRule;
import tenqube.parser.model.Transaction;

import static com.tenqube.visual_third.manager.PrefManager.PARSING_RULE_VERSION;

class VisualApi {

    public static final String TAG = VisualApi.class.getSimpleName();

    public static VisualApi mInstance;

    private Context context;
    private VisualApiService visualApiService;
    private PrefManager prefManager;



    public static VisualApi getInstance(Context context){
        synchronized (VisualApi.class) {
            if(mInstance == null){
                mInstance = new VisualApi(context.getApplicationContext());
            }

        }
        return mInstance;
    }

    private VisualApi(Context context) {
        this.context = context;
        prefManager = PrefManager.getInstance(context);
    }

    private VisualApiService getApiService() {
        if (visualApiService == null) {
            visualApiService = provideRetrofit(VisualApiService.URL).create(VisualApiService.class);

        }
        return visualApiService;
    }

    private Retrofit provideRetrofit(String url) {
        return new Retrofit.Builder()
                .baseUrl(url)
                .client(provideOkHttpClient())
                .addConverterFactory(GsonConverterFactory.create(new Gson()))
                .build();
    }

    private OkHttpClient provideOkHttpClient() {
        OkHttpClient.Builder okhttpClientBuilder = new OkHttpClient.Builder();
        okhttpClientBuilder.connectTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.readTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.writeTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.addInterceptor(new AddHeaderInterceptor());

        okhttpClientBuilder.addInterceptor(new AuthorizationInterceptor(prefManager, this));

        return okhttpClientBuilder.build();
    }


    public void signUp(SignUpInfo signUpInfo) throws AuthException {

        try {
            Response<VisualToken> response = getApiService().signUp(signUpInfo).execute();
            if (response.isSuccessful()) {
                // login request succeed, new token generated
                VisualToken visualToken = response.body();
                if(visualToken != null) {
                    prefManager.saveStringValue(PrefManager.ACCESS_TOKEN, visualToken.getAccessToken());
                    prefManager.saveStringValue(PrefManager.REFRESH_TOKEN, visualToken.getRefreshToken());
                    return;
                }
            }

        } catch (IOException e) {
            e.printStackTrace();
        }
        throw new AuthException("signUp error");

    }

    public void renewToken(String refreshToken) throws AuthException {

        try {
            Response<VisualToken> response = getApiService().renewToken(refreshToken).execute();
            if (response.isSuccessful()) {
                // login request succeed, new token generated
                VisualToken visualToken = response.body();
                if(visualToken != null) {
                    prefManager.saveStringValue(PrefManager.ACCESS_TOKEN, visualToken.getAccessToken());
                    prefManager.saveStringValue(PrefManager.REFRESH_TOKEN, visualToken.getRefreshToken());
                    return;
                }
            }

        } catch (IOException e) {
            e.printStackTrace();
        }
        throw new AuthException("RenewToken error");
    }


    public ParsingRule syncParsingRule() {
        try {
            Response<ParsingRule> response = getApiService().syncParsingRule(prefManager.loadIntValue(PARSING_RULE_VERSION, -1)).execute();
            if(response.isSuccessful()) {
                ParsingRule parsingRule = response.body();
                if(parsingRule != null)
                    prefManager.saveIntValue(PARSING_RULE_VERSION, parsingRule.ruleVersion);
                return response.body();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    public TranCompany sendToServerTransactionsWithSync(ArrayList<Transaction> transactions) {
        try {
            Response<TranCompany> response = getApiService().searchCompany(toVisualTransactions(transactions)).execute();
            if(response.isSuccessful()) {
                return response.body();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;

    }

    public void sendToServerTransactionsWithAsync(ArrayList<Transaction> transactions, Callback<TranCompany> callback) {

        Call<TranCompany> call = getApiService().searchCompany(toVisualTransactions(transactions));
        call.enqueue(callback);

    }


    private ArrayList<VisualTransaction> toVisualTransactions(ArrayList<Transaction> transactions) {
        ArrayList<VisualTransaction> visualTransactions = new ArrayList<>();
        for (Transaction transaction : transactions) {
            if(transaction != null)
                visualTransactions.add(new VisualTransaction(transaction));
        }
        return visualTransactions;
    }



}
