package com.tenqube.visual_third.web;

import android.content.Intent;
import android.support.v4.content.LocalBroadcastManager;
import android.webkit.JavascriptInterface;
import android.webkit.WebView;

import com.tenqube.visual_third.analysis.AnalysisService;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.js.Cards;
import com.tenqube.visual_third.model.js.CategoryInfo;
import com.tenqube.visual_third.model.js.Lv0Info;
import com.tenqube.visual_third.model.js.TransactionByIdsRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.js.Transactions;
import com.tenqube.visual_third.model.js.UpdateTransactionRequest;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.VisualWebActivity;

import java.util.ArrayList;
import java.util.Calendar;

import static com.tenqube.visual_third.parser.loader.OneLoader.ACTION_TRANSACTION_RECEIVED;
import static com.tenqube.visual_third.util.Mapper.toLv0s;
import static com.tenqube.visual_third.util.Utils.fromJson;
import static com.tenqube.visual_third.util.Utils.isEmpty;
import static com.tenqube.visual_third.util.Validator.isStr;

public class RepoImpl extends Base implements VisualInterface.Repo {


    private final VisualRepository repository;
    private final VisualApi api;
    private final AnalysisService analysisService;

    public RepoImpl(VisualWebActivity activity, WebView webView, VisualInterface.Error error,  VisualRepository repository, VisualApi api, AnalysisService service) {
        super(activity, webView, error);
        this.repository = repository;
        this.api = api;
        this.analysisService = service;
    }

    /**
     * 카테고리 목록 불러오기
     * @param callback 콜백 함수명
     */
    @JavascriptInterface
    public void getCategories(final String callback) {
        final String funcName = "getCategories";

        //start query;
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    isStr(callback, 100);

                    ArrayList<CategoryInfo.ServerCategory> serverCategories = repository.loadCategories();
                    ArrayList<CategoryInfo.UserCategory> userCategories = repository.loadUserCategories();
                    CategoryInfo categoryInfo = new CategoryInfo(serverCategories, userCategories);
                    callback(callback, categoryInfo);

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    e.printStackTrace();
                    onError(funcName, e.toString());

                }
            }
        }).start();

    }

    /**
     * 카드 목록 불러오기
     * @param callback 콜백 함수명
     */
    @JavascriptInterface
    public void getCards(final String callback) {
        final String funcName = "getCards";

        new Thread(new Runnable() {
            @Override
            public void run() {

                try {
                    isStr(callback, 100);

                    //start query;
                    ArrayList<Cards.Card> cardList = repository.loadCards();
                    Cards cards = new Cards(cardList);
                    callback(callback, cards);

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    e.printStackTrace();
                    onError(funcName, e.toString());
                }

            }
        }).start();

    }

    /**
     * 조건에 맞는 내역 목록 가져오기

     * @param params {
            year: 2015,
            month: 8,
            before: 3,
            callbackJS: 'callbackTransData'
        }
     */
    @JavascriptInterface
    public void getTransactions(final String params) {

        final String funcName = "getTransactions";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    TransactionRequest transactionRequest = fromJson(params, TransactionRequest.class);
                    transactionRequest.checkParams();
                    // start query
                    ArrayList<Transactions.Transaction> tranList = repository.loadTransactions(transactionRequest);
                    Transactions transactions = new Transactions(tranList);
                    callback(transactionRequest.getCallbackJS(), transactions);
                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    e.printStackTrace();
                    onError(funcName, e.toString());
                }

            }
        }).start();
    }

    /**
     * 조건에 맞는 내역 목록 가져오기

     * @param params {
    tranIds: [1,2,3],
    callbackJS: 'callbackTransData'
    }
     */
    @JavascriptInterface
    public void getTransactionsByIds(final String params) {

        final String funcName = "getTransactionsByIds";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    TransactionByIdsRequest transactionRequest = fromJson(params, TransactionByIdsRequest.class);
                    transactionRequest.checkParams();

                    // start query
                    ArrayList<Transactions.Transaction> tranList = repository.loadTransactions(transactionRequest);
                    Transactions transactions = new Transactions(tranList);
                    callback(transactionRequest.getCallbackJS(), transactions);
                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    e.printStackTrace();
                    onError(funcName, e.toString());
                }

            }
        }).start();
    }

    /**
     * 레벨 0 목록 가져오기
     * @param callback 콜백 함수명
     */
    @JavascriptInterface
    public void getLv0Contents(final String callback) {

        final String funcName = "getLv0Contents";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    isStr(callback, 100);

                    Calendar calendar = Calendar.getInstance();
                    int year = calendar.get(Calendar.YEAR);
                    int month = calendar.get(Calendar.MONTH) + 1;

                    TransactionRequest request = new TransactionRequest(year, month, 3, "");
                    ArrayList<com.tenqube.visual_third.model.analysis.Transaction> transactions = repository.loadAnalysisTransactions(request);
                    if(!isEmpty(transactions)) {
                        ArrayList<Analysis> analyses = analysisService.loadAnalysisList(transactions);
                        ArrayList<Lv0Info.Lv0> lv0s = toLv0s(analyses);

                        Lv0Info lv0Info = new Lv0Info(lv0s);
                        callback(callback, lv0Info);
                    }

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    e.printStackTrace();
                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    /**
     * 내역 업데이트
     * @param params {
            lCode: Number, // 대분류 코드
            mCode : Number, // 중분류 코드
            cateConfigId: Number, // 카테고리 설정 코드
            keyword: String, // 키워드
            amount: Number, // 금액
            cardId: Number, // 카드 아이디
            installmentCnt: Number, // 할부
            date: String, // 결제일
            memo: String, // 메모
            isExpense: Boolean, // 지출내역인지
            isAll: Boolean, // 모든 카테고리에 일괄적용할지
            callbackJS: 'callbackUpdateTrans' // 콜백
        }
     */
    @JavascriptInterface
    public void updateTransaction(final String params) {

        final String funcName = "updateTransaction";

        new Thread(new Runnable() {
            @Override
            public void run() {
                UpdateTransactionRequest updateTransactionRequest = null;
                try {
                    // 파싱
                    updateTransactionRequest = fromJson(params, UpdateTransactionRequest.class);

                    updateTransactionRequest.checkParams();

                    // 로컬 정보 저장
                    repository.updateTransaction(updateTransactionRequest);

                    // 검색 서버 요청
                    JoinedTransaction joinedTransaction = repository.loadJoinedTransaction(updateTransactionRequest.getTranId());
                    SearchCompanyRequest.Transaction transaction = new SearchCompanyRequest.Transaction(joinedTransaction);

                    transaction.setlCode(updateTransactionRequest.getlCode());
                    transaction.setmCode(updateTransactionRequest.getmCode());

                    ArrayList<SearchCompanyRequest.Transaction> transactions = new ArrayList<>();
                    transactions.add(transaction);
                    SearchCompanyResponse response = api.searchCompany(new SearchCompanyRequest(transactions));

                    // 요청된 정보 업데이트
                    if(response != null && response.getResults().size() > 0) {
                        repository.updateTransaction(response.getResults().get(0), updateTransactionRequest.isAll());
                        api.syncTransactions(null);
                    }

                    callback(updateTransactionRequest.getCallbackJS(), true);

                    // 업데이트됨
                    LocalBroadcastManager.getInstance(activity).sendBroadcast(new Intent(ACTION_TRANSACTION_RECEIVED));
                } catch (ParameterException e) {
                    callback(updateTransactionRequest.getCallbackJS(), false);
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    callback(updateTransactionRequest.getCallbackJS(), false);
                    e.printStackTrace();
                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    @JavascriptInterface
    public boolean shouldShowTranPopup() {
        return repository.shouldShowTranPopup();
    }

    @JavascriptInterface
    public void setTranPopup(boolean shouldShow) {
        repository.setTranPopup(shouldShow);
    }



}
