package com.tenqube.visual_third.web;

import android.webkit.JavascriptInterface;
import android.webkit.WebView;

import com.tenqube.visual_third.Callback;
import com.tenqube.visual_third.analysis.AnalysisService;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.entity.VisualNotification;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.js.AddPaymentRequest;
import com.tenqube.visual_third.model.js.AddPaymentResponse;
import com.tenqube.visual_third.model.js.Cards;
import com.tenqube.visual_third.model.js.CategoryInfo;
import com.tenqube.visual_third.model.js.DeleteTranRequest;
import com.tenqube.visual_third.model.js.InsertTransactionRequest;
import com.tenqube.visual_third.model.js.Lv0Info;
import com.tenqube.visual_third.model.js.SaveStartDayRequest;
import com.tenqube.visual_third.model.js.SettingNotiRequest;
import com.tenqube.visual_third.model.js.SettingNotiResponse;
import com.tenqube.visual_third.model.js.Success;
import com.tenqube.visual_third.model.js.TransactionByIdsRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.js.Transactions;
import com.tenqube.visual_third.model.js.UpdateTransactionRequest;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.VisualWebActivity;

import java.util.ArrayList;
import java.util.Calendar;

import tenqube.parser.constants.Constants;

import static com.tenqube.visual_third.util.Mapper.toLv0s;
import static com.tenqube.visual_third.util.Utils.fromJson;
import static com.tenqube.visual_third.util.Utils.isEmpty;
import static com.tenqube.visual_third.util.Validator.isStr;
import static com.tenqube.visual_third.util.Validator.notNull;

public class RepoImpl extends Base implements VisualInterface.Repo {

    private final VisualRepository repository;
    private final AnalysisService analysisService;
    private final VisualAlarmManager alarmManager;

    public RepoImpl(VisualWebActivity activity,
                    WebView webView,
                    VisualInterface.Error error,
                    VisualRepository repository,
                    AnalysisService service,
                    VisualAlarmManager alarmManager) {

        super(activity, webView, error);
        this.repository = repository;
        this.analysisService = service;
        this.alarmManager = alarmManager;
    }

    @JavascriptInterface
    public String getSyncStatus() {
        return repository.getSyncStatus();
    }

    @JavascriptInterface
    public void saveStartDay(final String params) {
        final String funcName = "saveStartDay";

        new Thread(new Runnable() {
            @Override
            public void run() {
                SaveStartDayRequest saveStartDayRequest = null;
                try {

                    saveStartDayRequest = fromJson(params, SaveStartDayRequest.class);
                    saveStartDayRequest.checkParams();

                    repository.saveStartDay(saveStartDayRequest.getStartDay());

                    callback(saveStartDayRequest.getCallbackJS(), new Success(true));
                } catch (ParameterException e) {
                    callback(saveStartDayRequest.getCallbackJS(),  new Success(false));
                    onError(funcName, e.toString());
                } catch (Exception e) {
                    onError(funcName, e.toString());
                }
            }
        }).start();
    }

    @JavascriptInterface
    public int getStartDay() {
        return repository.getStartDay();
    }

    @JavascriptInterface
    public void deleteTransactions(String params) {
        String funcName = "deleteTransactions";

        DeleteTranRequest deleteTranRequest = null;
        try {
            deleteTranRequest = fromJson(params, DeleteTranRequest.class);
            deleteTranRequest.checkParams();

            repository.updateDeletedTransactionById(deleteTranRequest.getIds());

            repository.syncTransactions(null);

            // 업데이트됨
            AnalysisServiceImpl.shouldRefresh = true;

            callback(deleteTranRequest.getCallbackJS(), new Success(true));

        } catch (Exception e) {
            if(deleteTranRequest != null) callback(deleteTranRequest.getCallbackJS(), new Success(false));
            onError(funcName, e.toString());
        }
    }

    /**
     * 카테고리 목록 불러오기
     * @param callback 콜백 함수명
     */
    @JavascriptInterface
    public void getCategories(final String callback) {
        final String funcName = "getCategories";

        //start query;
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    isStr(callback, 100);

                    ArrayList<CategoryInfo.ServerCategory> serverCategories = repository.loadCategories();
                    ArrayList<CategoryInfo.UserCategory> userCategories = repository.loadUserCategories();
                    CategoryInfo categoryInfo = new CategoryInfo(serverCategories, userCategories);
                    callback(callback, categoryInfo);

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());

                }
            }
        }).start();

    }

    /**
     * 카드 목록 불러오기
     * @param callback 콜백 함수명
     */
    @JavascriptInterface
    public void getCards(final String callback) {
        final String funcName = "getCards";

        new Thread(new Runnable() {
            @Override
            public void run() {

                try {
                    isStr(callback, 100);

                    //start query;
                    ArrayList<Cards.Card> cardList = repository.loadCards();
                    Cards cards = new Cards(cardList);
                    callback(callback, cards);

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }

            }
        }).start();

    }

    /**
     * 조건에 맞는 내역 목록 가져오기

     * @param params {
            year: 2015,
            month: 8,
            before: 3,
            hasStartDay: true
            callbackJS: 'callbackTransData'
        }
     */
    @JavascriptInterface
    public void getTransactions(final String params) {

        final String funcName = "getTransactions";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    TransactionRequest transactionRequest = fromJson(params, TransactionRequest.class);
                    transactionRequest.checkParams();

                    transactionRequest.setDwType(-1);
                    transactionRequest.setExceptType(-1);


//                    Log.i("getTransactions", transactionRequest.toString());

                    // start query
                    ArrayList<Transactions.Transaction> tranList = repository.loadTransactions(transactionRequest);
//                    for(Transactions.Transaction tran : tranList) {
//                        Log.i("getTransactions", tran.getKeyword());
//                        Log.i("getTransactions", tran.getSpentDate());
//
//
//                    }
                    Transactions transactions = new Transactions(tranList);
                    callback(transactionRequest.getCallbackJS(), transactions);
                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }

            }
        }).start();
    }

    /**
     * 조건에 맞는 내역 목록 가져오기

     * @param params {
    tranIds: [1,2,3],
    callbackJS: 'callbackTransData'
    }
     */
    @JavascriptInterface
    public void getTransactionsByIds(final String params) {

        final String funcName = "getTransactionsByIds";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    TransactionByIdsRequest transactionRequest = fromJson(params, TransactionByIdsRequest.class);
                    transactionRequest.checkParams();

                    // start query
                    ArrayList<Transactions.Transaction> tranList = repository.loadTransactions(transactionRequest);
                    Transactions transactions = new Transactions(tranList);
                    callback(transactionRequest.getCallbackJS(), transactions);
                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }

            }
        }).start();
    }

    /**
     * 레벨 0 목록 가져오기
     * @param callback 콜백 함수명
     */
    @JavascriptInterface
    public void getLv0Contents(final String callback) {

        final String funcName = "getLv0Contents";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    isStr(callback, 100);

                    Calendar calendar = Calendar.getInstance();
                    int year = calendar.get(Calendar.YEAR);
                    int month = calendar.get(Calendar.MONTH) + 1;

                    TransactionRequest request = new TransactionRequest(year, month, 3, true,"");
                    request.setDwType(Constants.DWType.WITHDRAW.ordinal());
                    request.setExceptType(0);

                    ArrayList<com.tenqube.visual_third.model.analysis.Transaction> transactions = repository.loadAnalysisTransactions(request);

//                    Log.i("loadAnalysisList", "transactions" + transactions.size());

                    Lv0Info lv0Info = new Lv0Info(new ArrayList<Lv0Info.Lv0>());
                    if(!isEmpty(transactions)) {
                        ArrayList<Analysis> analyses = analysisService.loadAnalysisList(transactions, repository.getStartDay());
                        ArrayList<Lv0Info.Lv0> lv0s = toLv0s(analyses);
                        lv0Info.setSuccess(lv0s);
                    }
                    callback(callback, lv0Info);

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    /**
     * 내역 추가
     * @param params {
    lCode: Number, // 대분류 코드
    mCode : Number, // 중분류 코드
    cateConfigId: Number, // 카테고리 설정 코드
    keyword: String, // 키워드
    amount: Number, // 금액
    cardId: Number, // 카드 아이디
    installmentCnt: Number, // 할부
    date: String, // 결제일
    memo: String, // 메모
    isExpense: Boolean, // 지출내역인지
    isAll: Boolean, // 모든 카테고리에 일괄적용할지
    callbackJS: 'callbackUpdateTrans' // 콜백
    }
     */
    @JavascriptInterface
    public void insertTransaction(final String params) {

        final String funcName = "insertTransaction";

        new Thread(new Runnable() {
            @Override
            public void run() {
                InsertTransactionRequest insertTransactionRequest = null;
                try {
                    // 파싱
                    insertTransactionRequest = fromJson(params, InsertTransactionRequest.class);

                    insertTransactionRequest.checkParams();

                    // 로컬 정보 저장
                    int id = repository.insertTransaction(insertTransactionRequest);
                    if(id == 0 ) {
                        callback(insertTransactionRequest.getCallbackJS(), new Success(false));
                        return;
                    }

                    insertTransactionRequest.setTranId(id);

                    // 검색 서버 요청
                    JoinedTransaction joinedTransaction = repository.loadJoinedTransaction(insertTransactionRequest.getTranId());
                    SearchCompanyRequest.Transaction transaction = new SearchCompanyRequest.Transaction(joinedTransaction);

                    transaction.setlCode(insertTransactionRequest.getlCode());
                    transaction.setmCode(insertTransactionRequest.getmCode());

                    ArrayList<SearchCompanyRequest.Transaction> transactions = new ArrayList<>();
                    transactions.add(transaction);
                    SearchCompanyResponse response = repository.searchCompany(new SearchCompanyRequest(transactions));

                    // 요청된 정보 업데이트
                    if(response != null && response.getResults().size() > 0) {
                        repository.updateTransaction(response.getResults().get(0), false);
                        repository.syncTransactions(null);
                    }

                    AnalysisServiceImpl.shouldRefresh = true;
                    callback(insertTransactionRequest.getCallbackJS(), new Success(true));
                } catch (ParameterException e) {
                    callback(insertTransactionRequest.getCallbackJS(), new Success(false));
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    /**
     * 내역 업데이트
     * @param params {
            lCode: Number, // 대분류 코드
            mCode : Number, // 중분류 코드
            cateConfigId: Number, // 카테고리 설정 코드
            keyword: String, // 키워드
            amount: Number, // 금액
            cardId: Number, // 카드 아이디
            installmentCnt: Number, // 할부
            date: String, // 결제일
            memo: String, // 메모
            isExpense: Boolean, // 지출내역인지
            isAll: Boolean, // 모든 카테고리에 일괄적용할지
            callbackJS: 'callbackUpdateTrans' // 콜백
        }
     */
    @JavascriptInterface
    public void updateTransaction(final String params) {

        final String funcName = "updateTransaction";

        new Thread(new Runnable() {
            @Override
            public void run() {
                UpdateTransactionRequest updateTransactionRequest = null;
                try {
                    // 파싱
                    updateTransactionRequest = fromJson(params, UpdateTransactionRequest.class);

                    updateTransactionRequest.checkParams();

                    // 로컬 정보 저장
                    repository.updateTransaction(updateTransactionRequest);

                    // 검색 서버 요청
                    JoinedTransaction joinedTransaction = repository.loadJoinedTransaction(updateTransactionRequest.getTranId());
                    SearchCompanyRequest.Transaction transaction = new SearchCompanyRequest.Transaction(joinedTransaction);

                    transaction.setlCode(updateTransactionRequest.getlCode());
                    transaction.setmCode(updateTransactionRequest.getmCode());

                    ArrayList<SearchCompanyRequest.Transaction> transactions = new ArrayList<>();
                    transactions.add(transaction);
                    SearchCompanyResponse response = repository.searchCompany(new SearchCompanyRequest(transactions));

                    // 요청된 정보 업데이트
                    if(response != null && response.getResults().size() > 0) {
                        repository.updateTransaction(response.getResults().get(0), updateTransactionRequest.isAll());
                        repository.syncTransactions(null);
                    }
                    AnalysisServiceImpl.shouldRefresh = true;

                    callback(updateTransactionRequest.getCallbackJS(), new Success(true));

                } catch (ParameterException e) {
                    callback(updateTransactionRequest.getCallbackJS(), new Success(false));
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    @JavascriptInterface
    public boolean shouldShowTranPopup() {
        return repository.isActiveTranPopup();
    }

    @JavascriptInterface
    public void setTranPopup(boolean shouldShow) {
        repository.setTranPopup(shouldShow);
    }

    @JavascriptInterface
    public void settingNotification(final String params) {
        final String funcName = "settingNotification";

        new Thread(new Runnable() {
            @Override
            public void run() {
                SettingNotiRequest settingNotiRequest = null;
                try {
                    // 파싱
                    settingNotiRequest = fromJson(params, SettingNotiRequest.class);

                    settingNotiRequest.checkParams();

                    // 디비 저장
                    repository.setActiveNoti(settingNotiRequest.getName().contains("weekly") ? "weekly" :settingNotiRequest.getName(),
                            settingNotiRequest.isEnabled());

                    if(settingNotiRequest.getHour() != null)
                        repository.updateNotiHour(settingNotiRequest.getName(), settingNotiRequest.getHour());

                    // 알림 정보 수정
                    alarmManager.setAlarms();

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();
    }

    @JavascriptInterface
    public void getNotificationSettings(final String callback) {
        final String funcName = "getNotificationSettings";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    notNull(callback);
                    // 로컬 정보 저장
                    ArrayList<VisualNotification> notifications = repository.loadNotifications();

                    if(isEmpty(notifications)) throw new ParameterException("empty");

                    ArrayList<SettingNotiRequest> settings = new ArrayList<>();
                    for(VisualNotification notification : notifications) {
                        settings.add(new SettingNotiRequest(notification.getName(), notification.isEnable(), notification.getHour()));
                    }

                    SettingNotiResponse res = new SettingNotiResponse(settings);
                    callback(callback, res);

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();
    }

    @JavascriptInterface
    public void initialize(final String callback) {
        final String funcName = "initialize";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    notNull(callback);
                    // 로컬 정보 저장

                    repository.initialize(new Callback<Boolean>() {
                        @Override
                        public void onDataLoaded(Boolean value) {
                            AnalysisServiceImpl.shouldRefresh = true;
                            callback(callback, new Success(value));
                        }
                    });

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    @JavascriptInterface
    public void signOut(final String callback) {
        final String funcName = "signOut";

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    notNull(callback);
                    // 로컬 정보 저장
                    repository.signOut(new Callback<Boolean>() {
                        @Override
                        public void onDataLoaded(Boolean value) {
                            AnalysisServiceImpl.shouldRefresh = true;
                            callback(callback, new Success(value));
                            activity.finish();
                        }
                    });

                } catch (ParameterException e) {
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();

    }

    @JavascriptInterface
    public void addPayment(final String params) {
        final String funcName = "addPayment";

        new Thread(new Runnable() {
            @Override
            public void run() {
                AddPaymentRequest addPaymentRequest = null;
                int id = -1;
                try {
                    // 파싱
                    addPaymentRequest = fromJson(params, AddPaymentRequest.class);

                    addPaymentRequest.checkParams();

                    // 로컬 정보 저장
                    id = repository.insertCard(addPaymentRequest.getName(), addPaymentRequest.getType());


                    callback(addPaymentRequest.getCallbackJS(), new AddPaymentResponse(id));
                } catch (ParameterException e) {
                    callback(addPaymentRequest.getCallbackJS(),  new AddPaymentResponse(id));
                    onError(funcName, e.toString());
                } catch (Exception e) {

                    onError(funcName, e.toString());
                }
            }
        }).start();
    }

}
